/********************************************************************************************
* Twitch Broadcasting SDK
*
* This software is supplied under the terms of a license agreement with Twitch Interactive, Inc. and
* may not be copied or used except in accordance with the terms of that agreement
* Copyright (c) 2012-2016 Twitch Interactive, Inc.
*********************************************************************************************/

#include "internal/pch.h"
#include "internal/webcam/testvideocapturesystem.h"
#include "internal/webcam/testvideocapturedevice.h"


static ttv::cam::TestVideoCaptureSystem* sInstance = nullptr;


#pragma region External Test Functions

extern "C" TTVSDK_API void TestVideoCaptureSystem_PluginDevice(const utf8char* uniqueId, const utf8char* name, const TTV_WebCamDeviceCapability* capabilities, unsigned int capabilityCount, unsigned int color)
{
	if (sInstance == nullptr)
	{
		return;
	}

	sInstance->PluginDevice(uniqueId, name, capabilities, capabilityCount, color);
}


extern "C" TTVSDK_API void TestVideoCaptureSystem_UnplugDevice(unsigned int deviceIndex)
{
	if (sInstance == nullptr)
	{
		return;
	}

	sInstance->UnplugDevice(deviceIndex);
}

#pragma endregion



namespace ttv
{
	namespace cam
	{
		namespace TestSystemMessageType
		{
			enum Enum
			{
				PluginDevice = SystemMessageType::SystemCustomMessages,
				UnplugDevice,
			};
		}

		struct PluginDeviceSystemMessage : public SystemMessage
		{
			utf8char uniqueId[kMaxDeviceIdLength];
			utf8char name[kMaxDeviceNameLength];
			TTV_WebCamDeviceCapability* capabilities;
			unsigned int capabilityCount;
			unsigned int color;

			PluginDeviceSystemMessage(const utf8char* i, const utf8char* n, const TTV_WebCamDeviceCapability* cap, unsigned int capCount, unsigned int clr) 
			{ 
				type = static_cast<SystemMessageType::Enum>(TestSystemMessageType::PluginDevice);
				capabilityCount = capCount;
				color = clr;

				// copy fields
				strncpy(uniqueId, i, sizeof(uniqueId));
				strncpy(name, n, sizeof(name));
				capabilities = new TTV_WebCamDeviceCapability[capCount];
				memcpy(capabilities, cap, sizeof(TTV_WebCamDeviceCapability)*capCount);
			}

			~PluginDeviceSystemMessage()
			{
				delete [] capabilities;
			}
		};

		struct UnplugDeviceSystemMessage : public SystemMessage
		{
			int deviceIndex;

			UnplugDeviceSystemMessage(int i) 
			{ 
				type = static_cast<SystemMessageType::Enum>(TestSystemMessageType::UnplugDevice);
				deviceIndex = i;
			}
		};
	}
}


ttv::cam::TestVideoCaptureSystem::TestVideoCaptureSystem()
:	mNumInitialTestDevices(0)
,	mTestDeviceUniqueIds(nullptr)
,	mTestDeviceNames(nullptr)
,	mCapabilities(nullptr)
,	mCapabilityCount(0)
,	mTestDeviceColors(nullptr)
{
	// for external testing access
	if (sInstance == nullptr)
	{
		sInstance = this;
	}
}


ttv::cam::TestVideoCaptureSystem::~TestVideoCaptureSystem()
{
	// for external testing access
	if (sInstance == this)
	{
		sInstance = nullptr;
	}

	mNumInitialTestDevices = 0;
	mTestDeviceUniqueIds = nullptr;
	mTestDeviceNames = nullptr;
	mTestDeviceColors = nullptr;
	mCapabilities = nullptr;
	mCapabilityCount = 0;
}


void ttv::cam::TestVideoCaptureSystem::SetInitializationParameters(unsigned int numInitialTestDevices, const utf8char** uniqueIds, const utf8char** names, const TTV_WebCamDeviceCapability* capabilities, unsigned int capabilityCount, const unsigned int* colors)
{
	mNumInitialTestDevices = numInitialTestDevices;
	mTestDeviceUniqueIds = uniqueIds;
	mTestDeviceNames = names;
	mTestDeviceColors = colors;
	mCapabilities = capabilities;
	mCapabilityCount = capabilityCount;
}


void ttv::cam::TestVideoCaptureSystem::PluginDevice(const utf8char* uniqueId, const utf8char* name, const TTV_WebCamDeviceCapability* capabilities, unsigned int capabilityCount, unsigned int color)
{
	UNUSED(uniqueId);
	UNUSED(name);
	UNUSED(capabilities);
	UNUSED(capabilityCount);
	UNUSED(color);

	assert(mSystemThreadRunning);

	// send the request to the system thread
	std::shared_ptr<SystemMessage> msg( new PluginDeviceSystemMessage(uniqueId, name, capabilities, capabilityCount, color) );
	mToSystemQ.push(msg);
}


void ttv::cam::TestVideoCaptureSystem::UnplugDevice(unsigned int deviceIndex)
{
	UNUSED(deviceIndex);

	assert(mSystemThreadRunning);

	// send the request to the system thread
	std::shared_ptr<SystemMessage> msg( new UnplugDeviceSystemMessage(deviceIndex) );
	mToSystemQ.push(msg);
}


TTV_ErrorCode ttv::cam::TestVideoCaptureSystem::InitializeSystem(const InitializeSystemMessage* msg)
{
	UNUSED(msg);

	// setup the initial test devices
	for (unsigned int i=0; i<mNumInitialTestDevices; ++i)
	{
		PluginDevice(mTestDeviceUniqueIds[i], mTestDeviceNames[i], mCapabilities, mCapabilityCount, mTestDeviceColors[i]);
	}

	mTestDeviceUniqueIds = nullptr;
	mTestDeviceNames = nullptr;
	mTestDeviceColors = nullptr;
	mCapabilityCount = 0;
	mCapabilities = nullptr;

	return TTV_EC_SUCCESS;
}


TTV_ErrorCode ttv::cam::TestVideoCaptureSystem::ShutdownSystem(const ShutdownSystemMessage* msg)
{
	UNUSED(msg);

	// nothing special needs to be done here
	return TTV_EC_SUCCESS;
}


TTV_ErrorCode ttv::cam::TestVideoCaptureSystem::HandleCustomSystemMessage(std::shared_ptr<cam::SystemMessage> msg)
{
	switch ( static_cast<int>(msg->type) )
	{
		case TestSystemMessageType::PluginDevice:
		{
			PluginDeviceSystemMessage* pMessage = static_cast<cam::PluginDeviceSystemMessage*>(msg.get());

			// if it's already plugged in then ignore
			auto iter = FindDeviceInstance(mSystemDevices, pMessage->uniqueId);
			if (iter != mSystemDevices.end())
			{
				return TTV_EC_SUCCESS;
			}

			std::shared_ptr<VideoCaptureDevice> device( new TestVideoCaptureDevice(this) );
			mSystemDevices.push_back(device);

			TestVideoCaptureDevice* pTestDevice = static_cast<TestVideoCaptureDevice*>(device.get());
			pTestDevice->SetTestDeviceData(pMessage->uniqueId, pMessage->name, pMessage->capabilities, pMessage->capabilityCount, pMessage->color);

			(void)device->Initialize(mNextDeviceIndex);

			mNextDeviceIndex++;

			return TTV_EC_SUCCESS;
		}
		case TestSystemMessageType::UnplugDevice:
		{
			UnplugDeviceSystemMessage* pMessage = static_cast<cam::UnplugDeviceSystemMessage*>(msg.get());

			// if not plugged in then ignore
			auto iter = FindDeviceInstance(mSystemDevices, pMessage->deviceIndex);
			if (iter == mSystemDevices.end())
			{
				return TTV_EC_SUCCESS;
			}

			// request that the device shutdown
			auto device = *iter;
			(void)device->Shutdown();

			return TTV_EC_SUCCESS;
		}
		default:
		{
			return VideoCaptureSystem::HandleCustomSystemMessage(msg);
		}
	}
}


void ttv::cam::TestVideoCaptureSystem::Update()
{
	// do nothing
}
