/********************************************************************************************
* Twitch Broadcasting SDK
*
* This software is supplied under the terms of a license agreement with Twitch Interactive, Inc. and
* may not be copied or used except in accordance with the terms of that agreement
* Copyright (c) 2012-2016 Twitch Interactive, Inc.
*********************************************************************************************/

#include "internal/pch.h"
#include "internal/webcam/videocapturesystem.h"
#include "internal/webcam/videocapturedevice.h"
#include "internal/webcam/webcamformat.h"
#include <unordered_set>
#include <type_traits>


ttv::cam::VideoCaptureDevice::VideoCaptureDevice(VideoCaptureSystem* parentSystem)
:	mSystem(parentSystem)
,	mDeviceThreadRunning(false)
,	mStatus(TTV_WEBCAM_DEVICE_UNINITIALIZED)
,	mFrames()
,	mCaptureFrame(-1)
,	mReadFrame(-1)
,	mClientFrameLocked(false)
{
	memset(&mClientDeviceInfo, 0, sizeof(mClientDeviceInfo));
	mClientDeviceInfo.deviceIndex = 0;
	mClientDeviceInfo.status = TTV_WEBCAM_DEVICE_UNINITIALIZED;

	for (size_t i=0; i<mFrames.size(); ++i)
	{
		memset(&mFrames, 0, sizeof(mFrames[i]));
	}

	CreateMutex(mMutex);
}


//-----------------------------------------------------------------------
ttv::cam::VideoCaptureDevice::~VideoCaptureDevice()
{
	// clean up the device info
	if (mClientDeviceInfo.capabilityList.list != nullptr)
	{
		delete [] mClientDeviceInfo.capabilityList.list;
		mClientDeviceInfo.capabilityList.list = nullptr;
	}

	if (mDeviceThread)
	{
		mDeviceThread->Join();
	}
}


TTV_ErrorCode ttv::cam::VideoCaptureDevice::Initialize(const unsigned int deviceIndex)
{
	mClientDeviceInfo.deviceIndex = deviceIndex;
	mClientDeviceInfo.status = TTV_WEBCAM_DEVICE_UNINITIALIZED;
	mStatus = TTV_WEBCAM_DEVICE_UNINITIALIZED;

	// the rest of the mClientDeviceInfo info will be filled in by the implementation in InitializeDevice

	// pass the request to the device thread
	mToDeviceQ.push(std::static_pointer_cast<cam::DeviceMessage>(std::make_shared<InitializeDeviceMessage>(deviceIndex)));

	// start the device thread
	char buffer[64];
	sprintf(buffer, "VideoCaptureDevice %u", deviceIndex);
	StartDeviceThread(std::bind(&VideoCaptureDevice::ThreadProc, this), buffer);

	return TTV_EC_SUCCESS;
}


TTV_ErrorCode ttv::cam::VideoCaptureDevice::Shutdown()
{
	if (!mDeviceThreadRunning)
	{
		return TTV_EC_SUCCESS;
	}

	// pass the request to the device thread
	std::shared_ptr<cam::DeviceMessage> msg( new cam::ShutdownDeviceMessage() );
	mToDeviceQ.push(msg);

	return TTV_EC_SUCCESS;
}


void ttv::cam::VideoCaptureDevice::StartDeviceThread(ttv::ThreadProc proc, const char* name)
{
	assert(!mDeviceThreadRunning);

	mDeviceThreadRunning = true;

	TTV_ErrorCode ec = ttv::CreateThread(proc, name, mDeviceThread);
	assert(TTV_SUCCEEDED(ec) && mDeviceThread != nullptr);

	mDeviceThread->Run();
}


TTV_ErrorCode ttv::cam::VideoCaptureDevice::Start(int capabilityIndex, TTV_WebCamDeviceStatusCallback callback, void* userdata)
{
	// pass the request to the device thread
	std::shared_ptr<cam::DeviceMessage> msg( new cam::StartDeviceMessage(capabilityIndex, callback, userdata) );
	mToDeviceQ.push(msg);

	return TTV_EC_SUCCESS;
}


TTV_ErrorCode ttv::cam::VideoCaptureDevice::Stop(TTV_WebCamDeviceStatusCallback callback, void* userdata)
{
	// pass the request to the device thread
	std::shared_ptr<cam::DeviceMessage> msg( new cam::StopDeviceMessage(callback, userdata) );
	mToDeviceQ.push(msg);

	return TTV_EC_SUCCESS;
}


const TTV_WebCamFrame* ttv::cam::VideoCaptureDevice::LockFrame()
{
	assert(!mClientFrameLocked);
	(void)mMutex->Lock();

	// there is a new frame available for the client
	if (mReadFrame >= 0)
	{
		TTV_WebCamFrame* frame = &mFrames[mReadFrame];

		// mark the frame as checked
		mReadFrame = -1;
		mClientFrameLocked = true;

		return frame;
	}
	// new frame not available yet
	else
	{
		(void)mMutex->Unlock();
		return nullptr;
	}
}


void ttv::cam::VideoCaptureDevice::UnlockFrame()
{
	assert(mClientFrameLocked);
	if (mClientFrameLocked)
	{
		mClientFrameLocked = false;
		(void)mMutex->Unlock();
	}
}


void ttv::cam::VideoCaptureDevice::SwapFrames()
{
	(void)mMutex->Lock();
	
	mReadFrame = mCaptureFrame;
	mCaptureFrame = (~static_cast<unsigned int>(mCaptureFrame)) & 0x1u; // 0 -> 1, 1 -> 0

	(void)mMutex->Unlock();
}


void ttv::cam::VideoCaptureDevice::ClearFrames()
{
	(void)mMutex->Lock();
	mReadFrame = -1;
	mCaptureFrame = -1;
	(void)mMutex->Unlock();
}


TTV_ErrorCode ttv::cam::VideoCaptureDevice::GetFrame(void* buffer, unsigned int pitch)
{
	TTV_ErrorCode ret = TTV_EC_SUCCESS;

	const TTV_WebCamFrame* frame = LockFrame();
	if (frame)
	{
		// straight copy of image data
		if (frame->capability.isNative)
		{
			memcpy(buffer, frame->imageBuffer, frame->bufferSize);
		}
		// the 
		else
		{
			// get the parent capability
			const TTV_WebCamDeviceCapability& parentCapability = GetParentCapability(frame->capability);

			assert(parentCapability.resolution.width == frame->capability.resolution.width);
			assert(parentCapability.resolution.height == frame->capability.resolution.height);
			assert(parentCapability.isNative);

			// convert between image formats while returning the buffer
			ret = ConvertImageFormat(frame->capability.resolution.width, frame->capability.resolution.height, parentCapability.format, frame->imageBuffer, frame->capability.format, reinterpret_cast<uint8_t*>(buffer), pitch);
		}
		UnlockFrame();
	}
	else
	{
		ret = TTV_EC_WEBCAM_FRAME_NOT_AVAILABLE;
	}

	return ret;
}


void ttv::cam::VideoCaptureDevice::SendEventToClient(std::shared_ptr<ClientMessage> msg)
{
	mSystem->SendEventToClient(msg);
}


void ttv::cam::VideoCaptureDevice::SendEventToSystem(std::shared_ptr<SystemMessage> msg)
{
	mSystem->SendEventToSystem(msg);
}


void ttv::cam::VideoCaptureDevice::ProcessDeviceMessages()
{
	std::shared_ptr<cam::DeviceMessage> msg;

	// Process all messages until the device thread is allowed to shutdown
	while (mDeviceThreadRunning && mToDeviceQ.try_pop(msg))
	{
		switch (msg->type)
		{
			// shutdown the device
			case DeviceMessageType::Initialize:
			{
				const InitializeDeviceMessage* pMessage = static_cast<InitializeDeviceMessage*>(msg.get());
				TTV_ErrorCode ret = InitializeDevice(pMessage);

				if (TTV_SUCCEEDED(ret))
				{
					mClientDeviceInfo.status = TTV_WEBCAM_DEVICE_STOPPED;
					mStatus = TTV_WEBCAM_DEVICE_STOPPED;
				}

				// notify the system of device initialize
				std::shared_ptr<DeviceInitializedSystemMessage> initMsg( new DeviceInitializedSystemMessage(mClientDeviceInfo.deviceIndex, ret) );
				SendEventToSystem(initMsg);

				break;
			}
			// shutdown the device
			case DeviceMessageType::Shutdown:
			{
				const ShutdownDeviceMessage* pMessage = static_cast<ShutdownDeviceMessage*>(msg.get());

				// force a stop
				if (mStatus == TTV_WEBCAM_DEVICE_STARTED)
				{
					(void)StopDevice(nullptr);
				}

				// stop the thread from processing any more messages
				mDeviceThreadRunning = false;

				(void)ShutdownDevice(pMessage);

				break;
			}
			// start the device
			case DeviceMessageType::Start:
			{
				const StartDeviceMessage* pMessage = static_cast<StartDeviceMessage*>(msg.get());
				(void)StartDevice(pMessage);
				break;
			}
			// stop the device
			case DeviceMessageType::Stop:
			{
				const StopDeviceMessage* pMessage = static_cast<StopDeviceMessage*>(msg.get());
				(void)StopDevice(pMessage);
				break;
			}
			// the derived implementation may have custom messages
			default:
			{
				(void)HandleCustomDeviceMessage(msg);
				break;
			}
		}
	}
}


TTV_ErrorCode ttv::cam::VideoCaptureDevice::HandleCustomDeviceMessage(std::shared_ptr<cam::DeviceMessage> msg)
{
	// Lint gives "unreachable code" warning after assert(false)
	//lint -save
	//lint -e527
	assert(false);
	return TTV_EC_SUCCESS;
	//lint -restore
}



void ttv::cam::VideoCaptureDevice::ThreadProc()
{
	ttv::AutoTracer entryAndExitTrace( "TwitchSDK", TTV_ML_INFO, "TestVideoCaptureDevice::ThreadProc" );

	while (mDeviceThreadRunning)
	{
		// handle incoming requests
		ProcessDeviceMessages();

		if (!mDeviceThreadRunning)
		{
			break;
		}

		// do internal processing
		Update();

		// wait a little while
		Sleep(10);
	}

	assert(mStatus != TTV_WEBCAM_DEVICE_STARTED);

	// notify the system of device shutdown
	std::shared_ptr<DeviceShutdownSystemMessage> shutdownMsg( new DeviceShutdownSystemMessage(mClientDeviceInfo.deviceIndex, mStatus == TTV_WEBCAM_DEVICE_UNINITIALIZED) );
	SendEventToSystem(shutdownMsg);
}


void ttv::cam::VideoCaptureDevice::WaitForThreadShutdown()
{
	// wait for the device thread to finish
	if (mDeviceThread != nullptr)
	{
		if (mDeviceThread->Joinable())
		{
			mDeviceThread->Join();
		}
		mDeviceThread.reset();
	}
}


int ttv::cam::VideoCaptureDevice::FindCapability(unsigned int capabilityIndex) const
{
	for (size_t i=0; i<mClientDeviceInfo.capabilityList.count; ++i)
	{
		// the capability index is not simply the position in the list, it may be a composite value if it's a non-native format
		if (mClientDeviceInfo.capabilityList.list[i].capabilityIndex == capabilityIndex)
		{
			return int(i);
		}
	}

	return -1;
}


struct Equal_WebCamDeviceCapability
{
	bool operator()(const TTV_WebCamDeviceCapability& a, const TTV_WebCamDeviceCapability& b) const
	{
		if (a.frameRate != b.frameRate) return false;
		if (a.resolution.width != b.resolution.width || a.resolution.height != b.resolution.height) return false;
		
		// the rest of the parameters don't matter

		return true;
	}
};


struct Hash_WebCamDeviceCapability
{
	static const size_t bucket_size = 4;

	size_t operator()(const TTV_WebCamDeviceCapability& cap) const
	{
		// the only important format-independent aspects of a capability are resolution and framerate
		size_t h = cap.frameRate;
		h = ((h << 5) + h) + cap.resolution.width;
		h = ((h << 5) + h) + cap.resolution.height;

		return h;
	}
};


struct FormatPriority
{
	TTV_WebcamFormat format;
	int priority;
};

/**
 * The formats we know how to convert from.  The easier formats to convert to RGBA have highter indices in the list.
 */
static FormatPriority sSourceFormatPriorities[] = 
{
	{ TTV_WEBCAM_FORMAT_YUY2, 0 },
	{ TTV_WEBCAM_FORMAT_RGB565, 1 },
	{ TTV_WEBCAM_FORMAT_RGB555, 1 },
	{ TTV_WEBCAM_FORMAT_RGB24, 2 },

	{ TTV_WEBCAM_FORMAT_XRGB32, 999 },
	{ TTV_WEBCAM_FORMAT_ARGB32, 1000 },
};


/**
 * Determines the priority value for selecting a format to convert from.  If it's not in the format list then
 * there is currently no conversion from the given format to ARGB32.
 */
int FindFormatPriority(TTV_WebcamFormat format)
{
	for (size_t i=0; i<sizeof(sSourceFormatPriorities)/sizeof(sSourceFormatPriorities[0]); ++i)
	{
		if (sSourceFormatPriorities[i].format == format)
		{
			return sSourceFormatPriorities[i].priority;
		}
	}

	return -1;
}


void ttv::cam::VideoCaptureDevice::ExtendCapabilities(std::vector<TTV_WebCamDeviceCapability>& list)
{
	// find all the different resolutions and framerates
	std::unordered_set<TTV_WebCamDeviceCapability, Hash_WebCamDeviceCapability, Equal_WebCamDeviceCapability> pruned;
	
	// create a list of the distinct resolutions and framerates, independent of format
	for (size_t i=0; i<list.size(); ++i)
	{
		const TTV_WebCamDeviceCapability& src = list[i];

		// only look at formats we know how to convert from right now
		int srcPriority = FindFormatPriority(src.format);
		if (srcPriority < 0)
		{
			continue;
		}

		bool add = false;
		auto iter = pruned.find(src);

		// there was already a matching capability of a different format
		if (iter != pruned.end())
		{
			auto prev = *iter;
			int prevPriority = FindFormatPriority(prev.format);

			add = srcPriority > prevPriority;

			// remove the previous entry since it's more expensive to convert from
			if (add)
			{
				(void)pruned.erase(iter);
			}
		}
		// not yet a matching capability
		else
		{
			add = true;
		}

		// add the entry
		if (add)
		{
			TTV_WebCamDeviceCapability copy = src;
			(void)pruned.insert(copy);
		}
	}

	// go over the list of source capabilities and create converted non-native ones
	for (auto iter = pruned.begin(); iter != pruned.end(); ++iter)
	{
		const TTV_WebCamDeviceCapability& capability = *iter;

		// Currently we only support converting to ARGB32 so if we have a native capability that is ARGB32 then
		// there's no need to create a non-native one that converts to ARGB32 from itself
		if (capability.format == TTV_WEBCAM_FORMAT_ARGB32)
		{
			continue;
		}

		TTV_WebCamDeviceCapability copy;
		copy.capabilityIndex = (capability.capabilityIndex << 16) | static_cast<unsigned int>(list.size());

		copy.format = TTV_WEBCAM_FORMAT_ARGB32;
		copy.resolution = capability.resolution;
		copy.frameRate = capability.frameRate;
		copy.isTopToBottom = capability.isTopToBottom;
		copy.isNative = false;

		list.push_back(copy);
	}
}


const TTV_WebCamDeviceCapability& ttv::cam::VideoCaptureDevice::GetParentCapability(const TTV_WebCamDeviceCapability& capability)
{
	if (capability.isNative)
	{
		return capability;
	}
	else
	{
		return mClientDeviceInfo.capabilityList.list[ (capability.capabilityIndex >> 16) ];
	}
}
