/********************************************************************************************
* Twitch Broadcasting SDK
*
* This software is supplied under the terms of a license agreement with Twitch Interactive, Inc. and
* may not be copied or used except in accordance with the terms of that agreement
* Copyright (c) 2012-2016 Twitch Interactive, Inc.
*********************************************************************************************/

#include "internal/pch.h"
#include "internal/webcam/videocapturesystem.h"
#include "internal/webcam/videocapturedevice.h"


ttv::cam::VideoCaptureSystem::VideoCaptureSystem()
:	mInitialized(false)
,	mNextDeviceIndex(1)
,	mSystemThreadRunning(false)
{
	memset(&mInterruptCallbacks, 0, sizeof(mInterruptCallbacks));

	CreateMutex(mClientDeviceListMutex);
}


ttv::cam::VideoCaptureSystem::~VideoCaptureSystem()
{
	assert(mClientDevices.size() == 0);
	assert(mSystemDevices.size() == 0);
}


TTV_ErrorCode ttv::cam::VideoCaptureSystem::Initialize(const TTV_WebCamCallbacks& interruptCallbacks, TTV_WebCamInitializationCallback initCallback, void* userdata)
{
	if (mInitialized)
	{
		return TTV_EC_ALREADY_INITIALIZED;
	}

	mInterruptCallbacks = interruptCallbacks;

	// pass the request to the system thread
	std::shared_ptr<cam::SystemMessage> msg( new cam::InitializeSystemMessage(initCallback, userdata) );
	mToSystemQ.push(msg);

	// kick off the system thread
	StartSystemThread(std::bind(&VideoCaptureSystem::ThreadProc, this), "VideoCaptureSystem");
	
	return TTV_EC_SUCCESS;
}


TTV_ErrorCode ttv::cam::VideoCaptureSystem::Shutdown(TTV_WebCamShutdownCallback callback, void* userdata)
{
	if (!mInitialized)
	{
		return TTV_EC_NOT_INITIALIZED;
	}

	// pass the request to the system thread
	std::shared_ptr<cam::SystemMessage> msg( new cam::ShutdownSystemMessage(callback, userdata) );
	mToSystemQ.push(msg);

	return TTV_EC_SUCCESS;
}


TTV_ErrorCode ttv::cam::VideoCaptureSystem::Start(unsigned int deviceIndex, unsigned int capabilityIndex, TTV_WebCamDeviceStatusCallback callback, void* userdata)
{
	if (!mInitialized)
	{
		return TTV_EC_NOT_INITIALIZED;
	}

	// pass the request to the system thread
	std::shared_ptr<cam::SystemMessage> msg( new cam::StartDeviceSystemMessage(deviceIndex, capabilityIndex, callback, userdata) );
	mToSystemQ.push(msg);

	return TTV_EC_SUCCESS;
}


TTV_ErrorCode ttv::cam::VideoCaptureSystem::Stop(unsigned int deviceIndex, TTV_WebCamDeviceStatusCallback callback, void* userdata)
{
	if (!mInitialized)
	{
		return TTV_EC_NOT_INITIALIZED;
	}

	// pass the request to the system thread
	std::shared_ptr<cam::SystemMessage> msg( new cam::StopDeviceSystemMessage(deviceIndex, callback, userdata) );
	mToSystemQ.push(msg);

	return TTV_EC_SUCCESS;
}


void ttv::cam::VideoCaptureSystem::StartSystemThread(ttv::ThreadProc proc, const char* name)
{
	assert(!mSystemThreadRunning);

	mSystemThreadRunning = true;

	TTV_ErrorCode ec = ttv::CreateThread(proc, name, mSystemThread);
	assert(TTV_SUCCEEDED(ec) && mSystemThread != nullptr);

	mSystemThread->Run();
}


void ttv::cam::VideoCaptureSystem::ProcessSystemMessages()
{
	std::shared_ptr<cam::SystemMessage> msg;

	while (mToSystemQ.try_pop(msg))
	{
		switch (msg->type)
		{
			// initialize the system
			case SystemMessageType::Initialize:
			{
				const InitializeSystemMessage* pMessage = static_cast<InitializeSystemMessage*>(msg.get());
	
				// make sure not already initialized
				assert(mSystemDevices.size() == 0);
				assert(mClientDevices.size() == 0);
				
				TTV_ErrorCode err = InitializeSystem(pMessage);

				// send initialized message to the main thread
				std::shared_ptr<ClientMessage> reply( new SystemInitializedClientMessage(pMessage->callback, err, pMessage->userdata) );
				GetClientQueue().push(reply);

				break;
			}
			// shutdown the system
			case SystemMessageType::Shutdown:
			{
				const ShutdownSystemMessage* pMessage = static_cast<ShutdownSystemMessage*>(msg.get());
	
				// not running
				if (!mSystemThreadRunning)
				{
					assert(mSystemDevices.size() == 0);
					break;
				}

				// shutdown all devices
				for (size_t i=0; i<mSystemDevices.size(); ++i)
				{
					std::shared_ptr<VideoCaptureDevice> device = mSystemDevices[i];
					(void)device->Shutdown();
				}

				// wait for all devices to be cleaned up
				while (mSystemDevices.size() > 0)
				{
					ProcessSystemMessages();
				}
				
				// call the implementation shutdown
				(void)ShutdownSystem(pMessage);

				// send a system shutdown message to the client thread
				std::shared_ptr<ClientMessage> reply( new SystemShutdownClientMessage(pMessage->callback, pMessage->userdata) );
				GetClientQueue().push(reply);

				mSystemThreadRunning = false;

				break;
			}
			// device initialized
			case SystemMessageType::DeviceInitialized:
			{
				const DeviceInitializedSystemMessage* pMessage = static_cast<DeviceInitializedSystemMessage*>(msg.get());
				(void)DeviceInitialized(pMessage);
				break;
			}
			// device shutdown
			case SystemMessageType::DeviceShutdown:
			{
				const DeviceShutdownSystemMessage* pMessage = static_cast<DeviceShutdownSystemMessage*>(msg.get());
				(void)DeviceShutdown(pMessage);
				break;
			}
			// start device
			case SystemMessageType::StartDevice:
			{
				const StartDeviceSystemMessage* pMessage = static_cast<StartDeviceSystemMessage*>(msg.get());
				auto device = FindDeviceInstance(mSystemDevices, pMessage->deviceIndex);
				if (device != mSystemDevices.end())
				{
					(void)(*device)->Start(pMessage->capabilityIndex, pMessage->callback, pMessage->userdata);
				}
				break;
			}
			// stop device
			case SystemMessageType::StopDevice:
			{
				const StopDeviceSystemMessage* pMessage = static_cast<StopDeviceSystemMessage*>(msg.get());
				auto device = FindDeviceInstance(mSystemDevices, pMessage->deviceIndex);
				if (device != mSystemDevices.end())
				{
					(void)(*device)->Stop(pMessage->callback, pMessage->userdata);
				}
				break;
			}
			// the derived implementation may have custom messages
			default:
			{
				(void)HandleCustomSystemMessage(msg);
				break;
			}
		}
	}
}


TTV_ErrorCode ttv::cam::VideoCaptureSystem::DeviceInitialized(const DeviceInitializedSystemMessage* msg)
{
	auto iter = FindDeviceInstance(mSystemDevices, msg->deviceIndex);
	assert(iter != mSystemDevices.end());

	if (iter == mSystemDevices.end())
	{
		return TTV_EC_WEBCAM_NOT_INITIALIZED;
	}

	auto device = *iter;

	// device initialized successfully
	if ( TTV_SUCCEEDED(msg->err) )
	{
		// notify the client
		std::shared_ptr<DeviceChangeClientMessage> reply( new DeviceChangeClientMessage(device, TTV_WEBCAM_DEVICE_FOUND, mInterruptCallbacks.deviceChangeCallback, mInterruptCallbacks.deviceChangeUserData) );
		mToClientQ.push(reply);
	}
	// device failed to initialize so leave it in the list but don't report it to the client
	else
	{
		// NOOP
	}
	
	return TTV_EC_SUCCESS;
}


TTV_ErrorCode ttv::cam::VideoCaptureSystem::DeviceShutdown(const DeviceShutdownSystemMessage* msg)
{
	auto iter = FindDeviceInstance(mSystemDevices, msg->deviceIndex);
	assert(iter != mSystemDevices.end());

	if (iter == mSystemDevices.end())
	{
		return TTV_EC_WEBCAM_NOT_INITIALIZED;
	}

	auto device = *iter;

	// wait for the device thread to shutdown
	device->WaitForThreadShutdown();
	
	// the last reference to the device will be released on the client thread in the handling of the TTV_WEBCAM_DEVICE_LOST event
	(void)mSystemDevices.erase(iter);

	// notify the client only if there wasn't an error during initialization
	if (!msg->initError)
	{
		std::shared_ptr<DeviceChangeClientMessage> reply( new DeviceChangeClientMessage(device, TTV_WEBCAM_DEVICE_LOST, mInterruptCallbacks.deviceChangeCallback, mInterruptCallbacks.deviceChangeUserData) );
		mToClientQ.push(reply);
	}

	return TTV_EC_SUCCESS;
}


TTV_ErrorCode ttv::cam::VideoCaptureSystem::HandleCustomSystemMessage(std::shared_ptr<cam::SystemMessage> msg)
{
	// Lint gives "unreachable code" warning after assert(false)
	//lint -save
	//lint -e527
	assert(false);
	return TTV_EC_SUCCESS;
	//lint -restore
}


void ttv::cam::VideoCaptureSystem::ThreadProc()
{
	ttv::AutoTracer entryAndExitTrace( "TwitchSDK", TTV_ML_INFO, "TestVideoCaptureSystem::ThreadProc" );

	while (mSystemThreadRunning)
	{
		// handle messages from the client and devices
		ProcessSystemMessages();

		if (!mSystemThreadRunning)
		{
			break;
		}

		// custom processing
		Update();

		// wait a little while
		Sleep(100);
	}
}


void ttv::cam::VideoCaptureSystem::FlushClientEvents()
{
	std::shared_ptr<cam::ClientMessage> msg;

	while (mToClientQ.try_pop(msg))
	{
		switch (msg->type)
		{
			// the system initialization result
			case ClientMessageType::SystemInitialized:
			{
				SystemInitializedClientMessage* pMessage = static_cast<cam::SystemInitializedClientMessage*>(msg.get());

				mInitialized = TTV_SUCCEEDED(pMessage->err);

				if (pMessage->callback != nullptr)
				{
					pMessage->callback(pMessage->err, pMessage->userdata);
				}

				break;
			}
			// the system shutdown result
			case ClientMessageType::SystemShutdown:
			{
				// the implementation should have cleaned up all devices by now by firing DeviceChange events
				assert(mClientDevices.size() == 0);

				SystemShutdownClientMessage* pMessage = static_cast<cam::SystemShutdownClientMessage*>(msg.get());

				mInitialized = false;

				// wait for the system thread to finish
				if (mSystemThread != nullptr)
				{
					if (mSystemThread->Joinable())
					{
						mSystemThread->Join();
					}
					mSystemThread.reset();
				}
				
				if (pMessage->callback != nullptr)
				{
					pMessage->callback(pMessage->err, pMessage->userdata);
				}

				break;
			}
			// device status
			case ClientMessageType::DeviceStatus:
			{
				DeviceStatusClientMessage* pMessage = static_cast<cam::DeviceStatusClientMessage*>(msg.get());

				(void)mClientDeviceListMutex->Lock();
				{
					// copy the status into the client device info
					auto iter = FindDeviceInstance(mClientDevices, pMessage->deviceIndex);
					assert(iter != mClientDevices.end());

					if (iter != mClientDevices.end())
					{
						auto device = *iter;
						device->GetClientDeviceInfo().status = pMessage->status;

						if (pMessage->callback != nullptr)
						{
							pMessage->callback(&device->GetClientDeviceInfo(), &pMessage->capability, pMessage->err, pMessage->userdata);
						}
					}
				}
				(void)mClientDeviceListMutex->Unlock();

				break;
			}
			// device change
			case ClientMessageType::DeviceChange:
			{
				DeviceChangeClientMessage* pMessage = static_cast<cam::DeviceChangeClientMessage*>(msg.get());

				switch (pMessage->change)
				{
					// add to the client device list
					case TTV_WEBCAM_DEVICE_FOUND:
					{
						(void)mClientDeviceListMutex->Lock();
						{
							// add to the client device list
							TTV_WebCamDevice& info = pMessage->device->GetClientDeviceInfo();
							auto device = FindDeviceInstance(mClientDevices, info.deviceIndex);

							assert(device == mClientDevices.end());
							if (device == mClientDevices.end())
							{
								mClientDevices.push_back(pMessage->device);
							}
						}
						(void)mClientDeviceListMutex->Unlock();

						break;
					}
					// remove from the client device list
					case TTV_WEBCAM_DEVICE_LOST:
					{
						(void)mClientDeviceListMutex->Lock();
						{
							// remove from the client device list
							TTV_WebCamDevice& info = pMessage->device->GetClientDeviceInfo();
							auto iter = FindDeviceInstance(mClientDevices, info.deviceIndex);

							assert(iter != mClientDevices.end());
							if (iter != mClientDevices.end())
							{
								(void)mClientDevices.erase(iter);
							}
						}
						(void)mClientDeviceListMutex->Unlock();

						break;
					}
					default:
						assert(false);
						break;
				}

				// notify the client
				if (mInterruptCallbacks.deviceChangeCallback != nullptr)
				{
					TTV_WebCamDevice& info = pMessage->device->GetClientDeviceInfo();
					mInterruptCallbacks.deviceChangeCallback(pMessage->change, &info, pMessage->err, mInterruptCallbacks.deviceChangeUserData);
				}

				break;
			}
			default:
			{
				assert(false);
				break;
			}
		}
	}

#if 0
	// go over each device and fire callbacks for new frames that are available
	if (mInterruptCallbacks.frameReadyCallback != nullptr)
	{
		for (size_t i=0; i<mClientDevices.size(); ++i)
		{
			std::shared_ptr<VideoCaptureDevice> device = mClientDevices[i];
			TTV_WebCamDevice& info = device->GetClientDeviceInfo();

			// not capturing so skip it
			if (info.status != TTV_WEBCAM_DEVICE_STARTED)
			{
				continue;
			}

			// check for a new frame and fire the callback if available
			const TTV_WebCamFrame* frame = device->LockFrame();
			if (frame)
			{
				mInterruptCallbacks.frameReadyCallback(frame, mInterruptCallbacks.frameReadyUserData);
				device->UnlockFrame();
			}
		}
	}
#endif
}


TTV_ErrorCode ttv::cam::VideoCaptureSystem::IsFrameAvailable(int deviceIndex, bool* available)
{
	TTV_ErrorCode ret = TTV_EC_SUCCESS;

	*available = false;

	(void)mClientDeviceListMutex->TryLock();
	{
		auto iter = FindDeviceInstance(mClientDevices, deviceIndex);
		if (iter == mClientDevices.end())
		{
			ret = TTV_EC_WEBCAM_DEVICE_NOT_FOUND;
		}
		else
		{
			auto device = *iter;
			*available = device->IsFrameAvailable();
		}
	}
	(void)mClientDeviceListMutex->Unlock();

	return ret;
}


TTV_ErrorCode ttv::cam::VideoCaptureSystem::GetFrame(int deviceIndex, void* buffer, unsigned int pitch)
{
	TTV_ErrorCode ret = TTV_EC_SUCCESS;

	(void)mClientDeviceListMutex->TryLock();
	{
		auto iter = FindDeviceInstance(mClientDevices, deviceIndex);
		if (iter == mClientDevices.end())
		{
			ret = TTV_EC_WEBCAM_DEVICE_NOT_FOUND;
		}
		else
		{
			auto device = *iter;
			ret = device->GetFrame(buffer, pitch);
		}
	}
	(void)mClientDeviceListMutex->Unlock();

	return ret;
}


std::vector< std::shared_ptr<ttv::cam::VideoCaptureDevice> >::iterator ttv::cam::VideoCaptureSystem::FindDeviceInstance(std::vector< std::shared_ptr<VideoCaptureDevice> >& list, unsigned int deviceIndex)
{
	for (auto iter = list.begin(); iter != list.end(); ++iter)
	{
		std::shared_ptr<VideoCaptureDevice> device = *iter;

		if (device->GetClientDeviceInfo().deviceIndex == deviceIndex)
		{
			return iter;
		}
	}

	return list.end();
}


std::vector< std::shared_ptr<ttv::cam::VideoCaptureDevice> >::iterator ttv::cam::VideoCaptureSystem::FindDeviceInstance(std::vector< std::shared_ptr<VideoCaptureDevice> >& list, const utf8char* uniqueId)
{
	for (auto iter = list.begin(); iter != list.end(); ++iter)
	{
		std::shared_ptr<VideoCaptureDevice> device = *iter;

		if ( strcmp(device->GetClientDeviceInfo().uniqueId, uniqueId) == 0 )
		{
			return iter;
		}
	}

	return list.end();
}


void ttv::cam::VideoCaptureSystem::SendEventToClient(std::shared_ptr<ClientMessage> msg)
{
	mToClientQ.push(msg);
}


void ttv::cam::VideoCaptureSystem::SendEventToSystem(std::shared_ptr<SystemMessage> msg)
{
	mToSystemQ.push(msg);
}
