/********************************************************************************************
* Twitch Broadcasting SDK
*
* This software is supplied under the terms of a license agreement with Twitch Interactive, Inc. and
* may not be copied or used except in accordance with the terms of that agreement
* Copyright (c) 2012-2016 Twitch Interactive, Inc.
*********************************************************************************************/

#include "internal/pch.h"
#include "twitchwebcam.h"
#include "internal/webcam/webcamapi.h"
#include "internal/webcam/videocapturesystem.h"
#include "internal/webcam/testvideocapturesystem.h"
#include "internal/webcam/webcamformat.h"

#if TTV_PLATFORM_WINDOWS
#	include "internal/webcam/win32/directshowvideocapturesystem.h"

#elif TTV_PLATFORM_IOS
#	include "internal/webcam/ios/iosvideocapturesystem.h"

#else
// TODO: implement other systems for other platforms

#endif



// External Test Access

static unsigned int sNumInitialTestDevices = 0;
static const utf8char** sTestDeviceUniqueIds = nullptr;
static const utf8char** sTestDeviceNames = nullptr;
static const TTV_WebCamDeviceCapability* sTestDeviceCapabilities = nullptr;
static unsigned int sTestDeviceCapabilityCount = 0;
static const unsigned int* sTestDeviceColors = nullptr;
static const char* sPreferredCaptureSystem = "";


extern "C" TTVSDK_API void WebcamAPI_SetPreferredCaptureSystem(const char* name)
{
	sPreferredCaptureSystem = name;
}

extern "C" TTVSDK_API void WebcamAPI_SetTestSystemParameters(unsigned int numInitialTestDevices, const utf8char** uniqueIds, const utf8char** names, const TTV_WebCamDeviceCapability* capabilities, unsigned int capabilityCount, const unsigned int* colors)
{
	sNumInitialTestDevices = numInitialTestDevices;
	sTestDeviceUniqueIds = uniqueIds;
	sTestDeviceNames = names;
	sTestDeviceCapabilities = capabilities;
	sTestDeviceCapabilityCount = capabilityCount;
	sTestDeviceColors = colors;
}



ttv::cam::WebcamAPI::WebcamAPI()
{
}


ttv::cam::WebcamAPI::~WebcamAPI()
{
}


TTV_ErrorCode ttv::cam::WebcamAPI::Init(const TTV_WebCamCallbacks* interruptCallbacks, TTV_WebCamInitializationCallback initCallback, void* userdata)
{
	if (interruptCallbacks == nullptr)
	{
		return TTV_EC_INVALID_ARG;
	}

	if (mVideoCaptureSystem != nullptr)
	{
		return TTV_EC_ALREADY_INITIALIZED;
	}

	if ( strcmp(sPreferredCaptureSystem, "") == 0 )
	{
#if TTV_PLATFORM_WINDOWS
		mVideoCaptureSystem = std::make_unique<DirectShowVideoCaptureSystem>();
#elif TTV_PLATFORM_IOS
		mVideoCaptureSystem = std::make_unique<IosVideoCaptureSystem>();
#else
		// TODO: implement other platforms
#endif
	}
	else if ( strcmp(sPreferredCaptureSystem, "TestVideoCaptureSystem") == 0 )
	{
		TestVideoCaptureSystem* pSystem = new TestVideoCaptureSystem();
		pSystem->SetInitializationParameters(sNumInitialTestDevices, sTestDeviceUniqueIds, sTestDeviceNames, sTestDeviceCapabilities, sTestDeviceCapabilityCount, sTestDeviceColors);
		mVideoCaptureSystem.reset(pSystem);
	}

	// no implementation found
	if (mVideoCaptureSystem == nullptr)
	{
		return TTV_EC_WEBCAM_NO_PLATFORM_SUPPORT;
	}

	TTV_ErrorCode err = mVideoCaptureSystem->Initialize(*interruptCallbacks, initCallback, userdata);
	return err;
}


TTV_ErrorCode ttv::cam::WebcamAPI::Shutdown(TTV_WebCamShutdownCallback callback, void* userdata)
{
	if (mVideoCaptureSystem == nullptr)
	{
		return TTV_EC_NOT_INITIALIZED;
	}

	// request that the system shutdown
	TTV_ErrorCode err = mVideoCaptureSystem->Shutdown(callback, userdata);
	
	// wait for the system to shutdown
	while (mVideoCaptureSystem->IsInitialized())
	{
		(void)FlushEvents();
		Sleep(10);
	}

	mVideoCaptureSystem.reset();

	return err;
}


TTV_ErrorCode ttv::cam::WebcamAPI::Start(unsigned int deviceIndex, unsigned int capabilityIndex, TTV_WebCamDeviceStatusCallback callback, void* userdata)
{
	if (mVideoCaptureSystem == nullptr)
	{
		return TTV_EC_NOT_INITIALIZED;
	}

	TTV_ErrorCode err = mVideoCaptureSystem->Start(deviceIndex, capabilityIndex, callback, userdata);
	return err;
}


TTV_ErrorCode ttv::cam::WebcamAPI::Stop(unsigned int deviceIndex, TTV_WebCamDeviceStatusCallback callback, void* userdata)
{
	if (mVideoCaptureSystem == nullptr)
	{
		return TTV_EC_NOT_INITIALIZED;
	}

	TTV_ErrorCode err = mVideoCaptureSystem->Stop(deviceIndex, callback, userdata);
	return err;
}


TTV_ErrorCode ttv::cam::WebcamAPI::FlushEvents()
{
	if (mVideoCaptureSystem == nullptr)
	{
		return TTV_EC_NOT_INITIALIZED;
	}

	mVideoCaptureSystem->FlushClientEvents();
	return TTV_EC_SUCCESS;
}


TTV_ErrorCode ttv::cam::WebcamAPI::IsFrameAvailable(int deviceIndex, bool* available)
{
	if (available == nullptr)
	{
		return TTV_EC_INVALID_ARG;
	}

	*available = false;

	TTV_ErrorCode err = mVideoCaptureSystem->IsFrameAvailable(deviceIndex, available);
	return err;
}


TTV_ErrorCode ttv::cam::WebcamAPI::GetFrame(int deviceIndex, void* buffer, unsigned int pitch)
{
	if (buffer == nullptr)
	{
		return TTV_EC_INVALID_ARG;
	}

	TTV_ErrorCode err = mVideoCaptureSystem->GetFrame(deviceIndex, buffer, pitch);
	return err;
}
