/********************************************************************************************
* Twitch Broadcasting SDK
*
* This software is supplied under the terms of a license agreement with Twitch Interactive, Inc. and
* may not be copied or used except in accordance with the terms of that agreement
* Copyright (c) 2012-2016 Twitch Interactive, Inc.
*********************************************************************************************/

#include "internal/pch.h"
#include "twitchwebcam.h"


namespace ttv
{
namespace cam
{

#define CLAMP_BYTE(x) (static_cast<unsigned char>(std::min(255, std::max(0, x))))


int GetPixelSize(TTV_WebcamFormat format)
{
	switch (format)
	{
		// TODO:
		case TTV_WEBCAM_FORMAT_RGB1:
		case TTV_WEBCAM_FORMAT_RGB4:
		case TTV_WEBCAM_FORMAT_RGB8:
			return 0;

		case TTV_WEBCAM_FORMAT_RGB555:
		case TTV_WEBCAM_FORMAT_RGB565:
		case TTV_WEBCAM_FORMAT_ARGB1555:
		case TTV_WEBCAM_FORMAT_ARGB4444:
		case TTV_WEBCAM_FORMAT_YUY2:
			return 2;

		case TTV_WEBCAM_FORMAT_RGB24:
			return 3;

		case TTV_WEBCAM_FORMAT_XRGB32:
		case TTV_WEBCAM_FORMAT_ARGB32:
		case TTV_WEBCAM_FORMAT_B10G10R10A2:
		case TTV_WEBCAM_FORMAT_R10G10B10A2:
			return 4;

		// TODO:
		case TTV_WEBCAM_FORMAT_AYUV:
		case TTV_WEBCAM_FORMAT_UYVY:
		case TTV_WEBCAM_FORMAT_IMC1:
		case TTV_WEBCAM_FORMAT_IMC2:
		case TTV_WEBCAM_FORMAT_IMC3:
		case TTV_WEBCAM_FORMAT_IMC4:
		case TTV_WEBCAM_FORMAT_YV12:
		case TTV_WEBCAM_FORMAT_NV12:
			return 0;

		default:
			return 0;
	}
}


int GetBufferSize(TTV_WebcamFormat format, int width, int height)
{
	switch (format)
	{
		// TODO:
		case TTV_WEBCAM_FORMAT_RGB1:
		case TTV_WEBCAM_FORMAT_RGB4:
		case TTV_WEBCAM_FORMAT_RGB8:
			return 0;

		case TTV_WEBCAM_FORMAT_RGB555:
		case TTV_WEBCAM_FORMAT_RGB565:
		case TTV_WEBCAM_FORMAT_ARGB1555:
		case TTV_WEBCAM_FORMAT_ARGB4444:
		case TTV_WEBCAM_FORMAT_RGB24:
		case TTV_WEBCAM_FORMAT_XRGB32:
		case TTV_WEBCAM_FORMAT_ARGB32:
		case TTV_WEBCAM_FORMAT_B10G10R10A2:
		case TTV_WEBCAM_FORMAT_R10G10B10A2:
		case TTV_WEBCAM_FORMAT_YUY2:
			return GetPixelSize(format) * width * height;

		// TODO:
		case TTV_WEBCAM_FORMAT_AYUV:
		case TTV_WEBCAM_FORMAT_UYVY:
		case TTV_WEBCAM_FORMAT_IMC1:
		case TTV_WEBCAM_FORMAT_IMC2:
		case TTV_WEBCAM_FORMAT_IMC3:
		case TTV_WEBCAM_FORMAT_IMC4:
		case TTV_WEBCAM_FORMAT_YV12:
		case TTV_WEBCAM_FORMAT_NV12:
			return 0;

		default:
			return 0;
	}
}








void CopyImageBuffer(unsigned int width, unsigned int height, uint8_t* srcBuffer, void* destBuffer, unsigned int destPitch, unsigned int pixelSize)
{
	// pitches match so do a straight memcpy
	if (destPitch == width*pixelSize)
	{
		memcpy(destBuffer, srcBuffer, width*height*pixelSize);
	}
	// pitches differ
	else
	{
		unsigned char* pDest = static_cast<unsigned char*>(destBuffer);
		unsigned char* pSrc = static_cast<unsigned char*>(srcBuffer);
		int rowSize = width*pixelSize;

		for (size_t y=0; y<height; ++y)
		{
			memcpy(pDest, pSrc, rowSize);

			pSrc += rowSize;
			pDest += destPitch;
		}
	}
}


void ConvertXRGB32ToARGB32(unsigned int width, unsigned int height, uint8_t* srcBuffer, void* destBuffer, unsigned int destPitch)
{
	unsigned char* pSrc = reinterpret_cast<unsigned char*>(srcBuffer);
	unsigned char* pLast = pSrc + width*height;
	unsigned char* pDest = static_cast<unsigned char*>(destBuffer);

	unsigned int x = 0;

	while (pSrc < pLast)
	{
		pDest[0] = pSrc[0];
		pDest[1] = pSrc[1];
		pDest[2] = pSrc[2];
		pDest[3] = 0xFF; // Ignore the X channel, which may be filled with garbage. Instead, fill with a solid alpha.

		pSrc += 4;
		pDest += 4;

		x++;
		if (x == width)
		{
			x = 0;
			pDest += destPitch;
		}
	}
}

void ConvertRGB555ToARGB32(unsigned int width, unsigned int height, uint8_t* srcBuffer, void* destBuffer, unsigned int destPitch)
{
	// TODO: implement this in SSE

	uint16_t* pSrc = reinterpret_cast<uint16_t*>(srcBuffer);
	uint16_t* pLast = pSrc + width*height;
	unsigned char* pDest = static_cast<unsigned char*>(destBuffer);

	unsigned int packedPitch = width * 4;
	unsigned int x = 0;

	while (pSrc < pLast)
	{
		pDest[0] = (pSrc[0] & 0x01F) << 3;
		pDest[1] = ((pSrc[0] >> 5) & 0x01F) << 3;
		pDest[2] = ((pSrc[0] >> 10) & 0x01F) << 3;
		pDest[3] = 0xFF;

		pSrc++;
		pDest += 4;

		// processing 2 pixels at a time
		x++;

		if (x == width)
		{
			x = 0;

			if (destPitch > packedPitch)
			{
				pDest += (destPitch - packedPitch);
			}
		}
	}
}

void ConvertRGB565ToARGB32(unsigned int width, unsigned int height, uint8_t* srcBuffer, void* destBuffer, unsigned int destPitch)
{
	// TODO: implement this in SSE

	uint16_t* pSrc = reinterpret_cast<uint16_t*>(srcBuffer);
	uint16_t* pLast = pSrc + width*height;
	unsigned char* pDest = static_cast<unsigned char*>(destBuffer);

	unsigned int packedPitch = width * 4;
	unsigned int x = 0;

	while (pSrc < pLast)
	{
		pDest[0] = (pSrc[0] & 0x01F) << 3;
		pDest[1] = ((pSrc[0] >> 5) & 0x03F) << 2;
		pDest[2] = ((pSrc[0] >> 11) & 0x01F) << 3;
		pDest[3] = 0xFF;

		pSrc++;
		pDest += 4;

		// processing 2 pixels at a time
		x++;

		if (x == width)
		{
			x = 0;

			if (destPitch > packedPitch)
			{
				pDest += (destPitch - packedPitch);
			}
		}
	}
}


void ConvertRGB24ToARGB32(unsigned int width, unsigned int height, uint8_t* srcBuffer, void* destBuffer, unsigned int destPitch)
{
	// TODO: implement this in SSE

	int numPixels = width * height;
	uint8_t* pSrc = static_cast<uint8_t*>(srcBuffer);
	uint8_t* pDest = static_cast<uint8_t*>(destBuffer);

	unsigned int x = 0;
	unsigned int packedPitch = width * 4;

	for (int i=0; i<numPixels; ++i)
	{
		pDest[0] = pSrc[0];
		pDest[1] = pSrc[1];
		pDest[2] = pSrc[2];
		pDest[3] = 0xFF;

		pSrc += 3;
		pDest += 4;

		x++;

		if (x == width)
		{
			x = 0;

			if (destPitch > packedPitch)
			{
				pDest += destPitch - packedPitch;
			}
		}
	}
}


void ConvertYUY2ToARGB32(unsigned int width, unsigned int height, uint8_t* srcBuffer, void* destBuffer, unsigned int destPitch)
{
	// TODO: implement this in SSE

	unsigned char* pSrc = srcBuffer;
	unsigned char* pLast = pSrc + GetBufferSize(TTV_WEBCAM_FORMAT_YUY2, width, height);
	unsigned char* pDest = static_cast<unsigned char*>(destBuffer);

	unsigned int packedPitch = width * 4;
	unsigned int x = 0;

	while (pSrc < pLast)
	{
		int d = pSrc[1] - 128;
		int e = pSrc[3] - 128;

		int c = pSrc[0] - 16;

		unsigned char r = CLAMP_BYTE(( 298*c + 409*e + 128 ) >> 8);
		unsigned char g = CLAMP_BYTE(( 298*c - 100*d - 208*e + 128 ) >> 8);
		unsigned char b = CLAMP_BYTE(( 298*c + 516*d + 128 ) >> 8);

		pDest[0] = b;
		pDest[1] = g;
		pDest[2] = r;
		pDest[3] = 0xFF;
		
		c = pSrc[2] - 16;

		r = CLAMP_BYTE(( 298*c + 409*e + 128 ) >> 8);
		g = CLAMP_BYTE(( 298*c - 100*d - 208*e + 128 ) >> 8);
		b = CLAMP_BYTE(( 298*c + 516*d + 128 ) >> 8);

		pDest[4] = b;
		pDest[5] = g;
		pDest[6] = r;
		pDest[7] = 0xFF;

		pSrc += 4;
		pDest += 8;

		// processing 2 pixels at a time
		x += 2;

		if (x >= width)
		{
			x = 0;

			if (destPitch > packedPitch)
			{
				pDest += (destPitch - packedPitch);
			}
		}
	}
}


void GenerateGarbageFrame(TTV_WebCamFrame* frame)
{
	for (size_t i=0; i<frame->bufferSize; ++i)
	{
		frame->imageBuffer[i] = static_cast<uint8_t>(rand());
	}
}


void GenerateGarbageFrame(uint8_t* buffer, unsigned int bufferSize)
{
	unsigned char* pDest = static_cast<unsigned char*>(buffer);

	for (size_t i=0; i<bufferSize; ++i)
	{
		pDest[i] = static_cast<uint8_t>(rand());
	}
}


TTV_ErrorCode ConvertToARGB(unsigned int width, unsigned int height, TTV_WebcamFormat srcFormat, uint8_t* srcBuffer, uint8_t* destBuffer, unsigned int destPitch)
{
	switch (srcFormat)
	{
		case TTV_WEBCAM_FORMAT_ARGB32:
		{
			CopyImageBuffer(width, height, srcBuffer, destBuffer, destPitch, 4);
			break;
		}
		case TTV_WEBCAM_FORMAT_XRGB32:
		{
			ConvertXRGB32ToARGB32(width, height, srcBuffer, destBuffer, destPitch);
			break;
		}
		case TTV_WEBCAM_FORMAT_RGB24:
		{
			ConvertRGB24ToARGB32(width, height, srcBuffer, destBuffer, destPitch);
			break;
		}
		case TTV_WEBCAM_FORMAT_RGB555:
		{
			ConvertRGB555ToARGB32(width, height, srcBuffer, destBuffer, destPitch);
			break;
		}
		case TTV_WEBCAM_FORMAT_RGB565:
		{
			ConvertRGB565ToARGB32(width, height, srcBuffer, destBuffer, destPitch);
			break;
		}
		case TTV_WEBCAM_FORMAT_YUY2:
		{
			ConvertYUY2ToARGB32(width, height, srcBuffer, destBuffer, destPitch);
			break;
		}
		default:
		{
			GenerateGarbageFrame(destBuffer, destPitch*height);
			return TTV_EC_WEBCAM_UNSUPPORTED_SOURCE_FORMAT;
		}
	}

	return TTV_EC_SUCCESS;
}


TTV_ErrorCode ConvertImageFormat(unsigned int width, unsigned int height, TTV_WebcamFormat srcFormat, uint8_t* srcBuffer, TTV_WebcamFormat destFormat, uint8_t* destBuffer, unsigned int destPitch)
{
	// Only ARGB32 is currently supported
	assert(destFormat == TTV_WEBCAM_FORMAT_ARGB32);

	if (destFormat != TTV_WEBCAM_FORMAT_ARGB32)
	{
		return TTV_EC_WEBCAM_UNSUPPORTED_TARGET_FORMAT;
	}

	return ConvertToARGB(width, height, srcFormat, srcBuffer, destBuffer, destPitch);
}

}
}
