/********************************************************************************************
* Twitch Broadcasting SDK
*
* This software is supplied under the terms of a license agreement with Twitch Interactive, Inc. and
* may not be copied or used except in accordance with the terms of that agreement
* Copyright (c) 2012-2016 Twitch Interactive, Inc.
*********************************************************************************************/

#include "internal/pch.h"
#include "internal/webcam/win32/directshowvideocapturedevice.h"
#include "internal/webcam/systemmessages.h"

namespace
{
	TTV_ErrorCode GetPins(IBaseFilter* const filter, const PIN_DIRECTION pinDirection, std::vector<IPin*>& pins);
	TTV_ErrorCode ConnectFirstPins(IBaseFilter* const filter1, const PIN_DIRECTION pinDirection1, IBaseFilter* const filter2, const PIN_DIRECTION pinDirection2, IGraphBuilder* const graph, const AM_MEDIA_TYPE* const mediaType);
	TTV_ErrorCode GetBufferSize(const TTV_WebCamDeviceCapability& mediaParameters, size_t& bufferSizeInBytes);
	TTV_ErrorCode CreateCOMObject(const GUID& classID, const IID& interfaceID, VOID** storageLocation);
	TTV_ErrorCode QueryCOMInterface(IUnknown* baseInterface, const IID& desiredInterface, void** storageLocation);
	TTV_ErrorCode AddFilterToGraph(const std::unique_ptr<IGraphBuilder, ttv::COMObjectDeleter<IGraphBuilder>>& graph, IBaseFilter* const source, std::wstring&& name);
	bool IsImageTopToBottom(const VIDEOINFOHEADER* const header, const TTV_WebcamFormat mediaFormat);
	TTV_WebcamFormat MediaTypeToWebcamFormat(AM_MEDIA_TYPE* const mediaType);

	
	// These functions are just to separate out the initialization and make it all easier to read
	TTV_ErrorCode CreateCaptureGraphBuilder(std::unique_ptr<ICaptureGraphBuilder2, ttv::COMObjectDeleter<ICaptureGraphBuilder2>>& captureGraphBuilder, const std::function<TTV_ErrorCode()>& nextStep);
	TTV_ErrorCode CreateGraph(std::unique_ptr<IGraphBuilder, ttv::COMObjectDeleter<IGraphBuilder>>& graph, std::unique_ptr<ICaptureGraphBuilder2, ttv::COMObjectDeleter<ICaptureGraphBuilder2>>& captureGraphBuilder, const std::function<TTV_ErrorCode()>& nextStep);
	TTV_ErrorCode CreateMediaControl(std::unique_ptr<IGraphBuilder, ttv::COMObjectDeleter<IGraphBuilder>>& graph, std::unique_ptr<IMediaControl, ttv::COMObjectDeleter<IMediaControl>>& mediaControl, const std::function<TTV_ErrorCode()>& nextStep);
	TTV_ErrorCode CreateMediaEvent(std::unique_ptr<IGraphBuilder, ttv::COMObjectDeleter<IGraphBuilder>>& graph, std::unique_ptr<IMediaEventEx, ttv::COMObjectDeleter<IMediaEventEx>>& event, const std::function<TTV_ErrorCode()>& nextStep);
	TTV_ErrorCode AddWebcamToGraph(std::unique_ptr<IGraphBuilder, ttv::COMObjectDeleter<IGraphBuilder>>& graph, std::shared_ptr<IBaseFilter>& source, const std::function<TTV_ErrorCode()>& nextStep);
	TTV_ErrorCode CreateStreamConfig(std::unique_ptr<ICaptureGraphBuilder2, ttv::COMObjectDeleter<ICaptureGraphBuilder2>>& captureGraphBuilder, std::shared_ptr<IBaseFilter>& source, std::unique_ptr<IAMStreamConfig, ttv::COMObjectDeleter<IAMStreamConfig>>& streamConfig, const std::function<TTV_ErrorCode()>& nextStep);
	TTV_ErrorCode SetCapability(AM_MEDIA_TYPE* const mediaType, std::unique_ptr<IAMStreamConfig, ttv::COMObjectDeleter<IAMStreamConfig>>& streamConfig, const std::function<TTV_ErrorCode()>& nextStep);
	TTV_ErrorCode AddFrameGrabberToGraph(std::unique_ptr<IBaseFilter, ttv::COMObjectDeleter<IBaseFilter>>& grabberFilter, std::unique_ptr<IGraphBuilder, ttv::COMObjectDeleter<IGraphBuilder>>& graph, std::shared_ptr<IBaseFilter>& source, const AM_MEDIA_TYPE* const mediaType, std::unique_ptr<ISampleGrabber, ttv::COMObjectDeleter<ISampleGrabber>>& sampleGrabber, const std::function<TTV_ErrorCode()>& nextStep);
	TTV_ErrorCode AddNullRendererToGraph(std::unique_ptr<IGraphBuilder, ttv::COMObjectDeleter<IGraphBuilder>>& graph, std::unique_ptr<IBaseFilter, ttv::COMObjectDeleter<IBaseFilter>>& grabberFilter, const AM_MEDIA_TYPE* const mediaType, const std::function<TTV_ErrorCode()>& nextStep);
	TTV_ErrorCode RunGraph(std::unique_ptr<IMediaControl, ttv::COMObjectDeleter<IMediaControl>>& mediaControl, const std::function<TTV_ErrorCode()>& nextStep);
	TTV_ErrorCode InitializeCaptureFrames(const TTV_WebCamDeviceCapability& capability, std::array<TTV_WebCamFrame, 2>& frames, volatile int& captureFrame, unsigned int deviceIndex, const std::function<TTV_ErrorCode()>& nextStep);
	TTV_ErrorCode StartCallbacks(ttv::cam::SampleGrabberCallback& callback, std::unique_ptr<ISampleGrabber, ttv::COMObjectDeleter<ISampleGrabber>>& sampleGrabber, const std::shared_ptr<ttv::cam::DirectShowVideoCaptureDevice>& that);

} // unnamed namespace


namespace
{

	TTV_ErrorCode GetPins(IBaseFilter* const filter, const PIN_DIRECTION pinDirection, std::vector<IPin*>& pins)
	{
		auto returning = TTV_EC_SUCCESS;

		IEnumPins* pinEnumerator = nullptr;
		HRESULT result = filter->EnumPins(&pinEnumerator);
		switch (result)
		{
		case S_OK: // Success
			break;
		case E_OUTOFMEMORY:
			returning = TTV_EC_WEBCAM_OUT_OF_MEMORY;
			break;
		case E_POINTER:
			// Should not happen
			returning = TTV_EC_WEBCAM_COULD_NOT_COMPLETE;
			break;
		default:
			returning = TTV_EC_WEBCAM_UNKNOWN_ERROR;
		}

		if (TTV_SUCCEEDED(returning) && pinEnumerator != nullptr)
		{
			bool morePinsLeft = true;

			do
			{
				IPin* pin = nullptr;
				result = pinEnumerator->Next(1, &pin, 0);

				switch (result)
				{
				case S_OK:
					break;
				case S_FALSE: // No more items in the enumerator
					morePinsLeft = false;
					break;
				default:
					returning = TTV_EC_WEBCAM_UNKNOWN_ERROR;
				}

				if (TTV_SUCCEEDED(returning) && pin != nullptr)
				{
					PIN_DIRECTION currentPinDirection;
					result = pin->QueryDirection(&currentPinDirection);

					//lint -save
					//lint -e1924   C-style cast
					// This is from the FAILED() macro, which is out of our control.
					if (FAILED(result))
					//lint -restore
					{
						// Expected error codes:
						// E_POINTER
						returning = TTV_EC_WEBCAM_COULD_NOT_COMPLETE;
					}

					if (TTV_SUCCEEDED(returning) && (currentPinDirection == pinDirection))
					{
						(void)pin->AddRef();
						pins.push_back(pin);
					}
				}

				if (pin != nullptr)
				{
					(void)pin->Release();
				}

			} while (morePinsLeft == true);
		}

		if (pinEnumerator != nullptr)
		{
			(void)pinEnumerator->Release();
		}

		return returning;
	}


	TTV_ErrorCode ConnectFirstPins(IBaseFilter* const filter1, const PIN_DIRECTION pinDirection1, IBaseFilter* const filter2, const PIN_DIRECTION pinDirection2, IGraphBuilder* const graph, const AM_MEDIA_TYPE* const mediaType)
	{
		std::vector<IPin*> filter1Pins;
		auto returning = ::GetPins(filter1, pinDirection1, filter1Pins);

		if (TTV_SUCCEEDED(returning))
		{
			std::vector<IPin*> filter2Pins;
			returning = ::GetPins(filter2, pinDirection2, filter2Pins);

			if (TTV_SUCCEEDED(returning))
			{
				auto result = graph->ConnectDirect(filter1Pins[0], filter2Pins[0], mediaType);

				//lint -save
				//lint -e1924 C-style cast
				// This is coming from things like S_OK, which are not mine to control.
				switch (result)
				{
				case S_OK: // success
					break;
				case E_FAIL:
					// fall through
				case E_INVALIDARG: // the docs don't mention that this error is possible, but apparently it is
					// fall through
				case E_NOTIMPL:
					// fall through
				case E_POINTER:
					// fall through
				case VFW_E_NOT_IN_GRAPH: // one of the pins is not in this graph
					// fall through
				case VFW_E_CIRCULAR_GRAPH:
					// fall through
				case VFW_E_ALREADY_CONNECTED: // One of the pins is already connected
					// fall through
				case VFW_E_NO_ACCEPTABLE_TYPES: // Cannot find an acceptable media type
					// fall through
				case VFW_E_NO_TRANSPORT: // Pins cannot agree on a transport, or there is no allocator for the connection
					// fall through
				case VFW_E_NOT_STOPPED: // The filter is active and the pin does not support dynamic reconnection.
					// fall through
				case VFW_E_TYPE_NOT_ACCEPTED: // The specified media type is not acceptable.
					returning = TTV_EC_WEBCAM_COULD_NOT_COMPLETE;
					break;
				case static_cast<HRESULT>(ERROR_NO_MATCH): // There was no match for the specified key in the index. 
					// Apparently, this happens for devices which do not support UVC1.1??
					// http://social.msdn.microsoft.com/Forums/en-US/mediafoundationdevelopment/thread/c2d3868a-92f0-43fa-9f01-96208292c495/
					returning = TTV_EC_WEBCAM_UNKNOWN_ERROR;
					break;
				case E_UNEXPECTED: // Not sure what is causing this
					returning = TTV_EC_WEBCAM_UNKNOWN_ERROR;
					break;
				default:
					returning = TTV_EC_WEBCAM_UNKNOWN_ERROR;
				}
				//lint -restore
			}

			for (auto currentPin = filter2Pins.begin(); currentPin != filter2Pins.end(); ++currentPin)
			{
				(void)(*currentPin)->Release();
			}
		}

		for (auto currentPin = filter1Pins.begin(); currentPin != filter1Pins.end(); ++currentPin)
		{
			(void)(*currentPin)->Release();
		}

		return returning;
	}


	//-V:GetBufferSize:112
	TTV_ErrorCode GetBufferSize(const TTV_WebCamDeviceCapability& mediaParameters, size_t& bufferSizeInBytes)
	{
		auto returning = TTV_EC_SUCCESS;

		const auto height = mediaParameters.resolution.height;
		const auto width = mediaParameters.resolution.width;
		const auto area = height * width;
		size_t bitsPerPixel = 0;
		size_t paletteSize = 0;

		const size_t BytesIn32Bits = 4; //-V112   Dangerous magic number 4 used     - No, I'm giving is a name and making it not a magic number.

		switch (mediaParameters.format)
		{
		case TTV_WEBCAM_FORMAT_RGB1: // 1 bit per pixel, palettized
			bitsPerPixel = 1;
			paletteSize = (1 << 1) * BytesIn32Bits /* assume each palette entry is 32-bit */;
			break;
		case TTV_WEBCAM_FORMAT_RGB4: // 4 bpp, palettized
			bitsPerPixel = 4; //-V112   Dangerous magic number 4 used
			paletteSize = (1 << 4) * BytesIn32Bits /* assume each palette entry is 32-bit */;
			break;
		case TTV_WEBCAM_FORMAT_RGB8: // 8 bpp, palettized
			bitsPerPixel = 8;
			paletteSize = (1 << 8) * BytesIn32Bits /* assume each palette entry is 32-bit */;
			break;
		case TTV_WEBCAM_FORMAT_RGB555: // 16 bpp, XRRRRRGG GGGBBBBB
			bitsPerPixel = 16;
			break;
		case TTV_WEBCAM_FORMAT_RGB565: // 16 bpp, RRRRRGGG GGGBBBBB
			bitsPerPixel = 16;
			break;
		case TTV_WEBCAM_FORMAT_RGB24: // 24 bpp, R8G8B8
			bitsPerPixel = 24;
			break;
		case TTV_WEBCAM_FORMAT_XRGB32: // 32 bpp, B8G8R8X8
			bitsPerPixel = 32; //-V112   Dangerous magic number 32 used.
			break;
		case TTV_WEBCAM_FORMAT_ARGB1555: // 16 bpp with 1 alpha, ARRRRRGG GGGBBBBB
			bitsPerPixel = 16;
			break;
		case TTV_WEBCAM_FORMAT_ARGB32: // 32 bpp with alpha - A8R8G8B8
			bitsPerPixel = 32; //-V112   Dangerous magic number 32 used.
			break;
		case TTV_WEBCAM_FORMAT_ARGB4444: // 16 bpp, 4 bits alpha
			bitsPerPixel = 16;
			break;
		case TTV_WEBCAM_FORMAT_B10G10R10A2: // 32 bpp, 10 RGB, 2 alpha, BBBBBBBB BBGGGGGG GGGGRRRR RRRRRRAA
			bitsPerPixel = 32; //-V112   Dangerous magic number 32 used.
			break;
		case TTV_WEBCAM_FORMAT_R10G10B10A2: // 32 bpp, 10 RGB, 2 alpha, RRRRRRRR RRGGGGGG GGGGBBBB BBBBBBAA
			bitsPerPixel = 32; //-V112   Dangerous magic number 32 used.
			break;
		case TTV_WEBCAM_FORMAT_AYUV: // AYUV 4:4:4 packed 8 bpp
			bitsPerPixel = 32; //-V112   Dangerous magic number 32 used.
			break;
		case TTV_WEBCAM_FORMAT_YUY2: // YUY2 4:2:2 packed 8 bpp
			bitsPerPixel = 16;
			break;
		case TTV_WEBCAM_FORMAT_UYVY: // UYVY 4:2:2 packed 8 bpp
			bitsPerPixel = 16;
			break;
		case TTV_WEBCAM_FORMAT_IMC1: // IMC1 4:2:0 planar 8 bpp
			bitsPerPixel = 12;
			break;
		case TTV_WEBCAM_FORMAT_IMC2: // IMC2 4:2:0 planar 8 bpp
			bitsPerPixel = 12;
			break;
		case TTV_WEBCAM_FORMAT_IMC3: // IMC3 4:2:0 planar 8 bpp
			bitsPerPixel = 12;
			break;
		case TTV_WEBCAM_FORMAT_IMC4: // IMC4 4:2:0 planar 8 bpp
			bitsPerPixel = 12;
			break;
		case TTV_WEBCAM_FORMAT_YV12: // YV12 4:2:0 planar 8 bpp
			bitsPerPixel = 12;
			break;
		case TTV_WEBCAM_FORMAT_NV12: // NV12 4:2:0 planar 8 bpp
			bitsPerPixel = 12;
			break;
		default:
			returning = TTV_EC_WEBCAM_INVALID_PARAMETER;
		}

		if (TTV_SUCCEEDED(returning))
		{
			bufferSizeInBytes = (static_cast<size_t>(area) * bitsPerPixel + 7) // adding 7 so sizes that do not align to a byte will bump up to the next byte when the remainder is trimmed off during the integer division
			                    / 8 + paletteSize;
		}

		return returning;
	}	


	void FillDeviceInfo(TTV_WebCamDevice& deviceInfo, const std::string& name, const std::string& uniqueIdentifier)
	{
		// copy the name
		if (name.size() >= sizeof(deviceInfo.name))
		{
			deviceInfo.name[0] = '\0';
			assert(false);
		}
		else
		{
			strncpy(deviceInfo.name, name.c_str(), sizeof(deviceInfo.name));
		}

		// copy the unique id
		if (uniqueIdentifier.size() >= sizeof(deviceInfo.uniqueId) )
		{
			deviceInfo.uniqueId[0] = '\0';
			assert(false);
		}
		else
		{
			strncpy(deviceInfo.uniqueId, uniqueIdentifier.c_str(), sizeof(deviceInfo.uniqueId));
		}
	}


	TTV_ErrorCode CreateCOMObject(const GUID& classID, const IID& interfaceID, VOID** storageLocation)
	{
		auto returning = TTV_EC_SUCCESS;

		HRESULT result = CoCreateInstance(classID, NULL, static_cast<DWORD>(CLSCTX_INPROC_SERVER), interfaceID, storageLocation);

		//lint -save
		//lint -e1924 C-style cast
		// This is coming from things like S_OK, which are not mine to control.
		switch (result)
		{
		case S_OK: // succes
			break;
		case REGDB_E_CLASSNOTREG: // A class matching the specified class is not registered. Or the server types in the registry are corrupt.
			// fall through
		case E_NOINTERFACE: // The specified class does not implement the requested interface.
			returning = TTV_EC_WEBCAM_NO_PLATFORM_SUPPORT;
			break;
		case CLASS_E_NOAGGREGATION:
			// fall through
		case E_POINTER:
			returning = TTV_EC_WEBCAM_COULD_NOT_COMPLETE;
			break;
		default:
			returning = TTV_EC_WEBCAM_UNKNOWN_ERROR;
		}
		//lint -restore

		assert( TTV_SUCCEEDED(returning) );

		return returning;
	}


	TTV_ErrorCode QueryCOMInterface(IUnknown* baseInterface, const IID& desiredInterface, void** storageLocation)
	{
		auto returning = TTV_EC_SUCCESS;

		auto result = baseInterface->QueryInterface(desiredInterface, storageLocation);
							
		//lint -save
		//lint -e1924 C-style cast
		// This is coming from things like S_OK, which are not mine to control.
		switch (result)
		{
		case S_OK: // success
			break;
		case E_NOINTERFACE: // Interface not found
			// fall through
		case E_POINTER:
			// should not happen
			returning = TTV_EC_WEBCAM_COULD_NOT_COMPLETE;
			break;
		default:
			returning = TTV_EC_WEBCAM_UNKNOWN_ERROR;
		}
		//lint -restore

		assert( TTV_SUCCEEDED(returning) );

		return returning;
	}


	TTV_ErrorCode AddFilterToGraph(const std::unique_ptr<IGraphBuilder, ttv::COMObjectDeleter<IGraphBuilder>>& graph, IBaseFilter* const source, std::wstring&& name)
	{
		auto returning = TTV_EC_SUCCESS;

		auto result = graph->AddFilter(source, name.c_str());
								
		//lint -save
		//lint -e1924 C-style cast
		// This is coming from things like S_OK, which are not mine to control.
		switch (result)
		{
		case S_OK: // success
			// fall through
		case VFW_S_DUPLICATE_NAME: // Name already in use, but succeeded anyway (by automatically modifying the name to attempt to have a unique name)
			break;
		case VFW_E_DUPLICATE_NAME: // Name already in use, caused failure (because it could not automatically modify the name to become unique)
			// should not happen
			// fall through
		case E_POINTER:
			// should not happen
			// fall through
		case VFW_E_CERTIFICATION_FAILURE: // this filter requires a software key
			// fall through
		case E_FAIL:
			returning = TTV_EC_WEBCAM_COULD_NOT_COMPLETE;
			break;
		case E_OUTOFMEMORY:
			returning = TTV_EC_WEBCAM_OUT_OF_MEMORY;
			break;
		default:
			returning = TTV_EC_WEBCAM_UNKNOWN_ERROR;
		}
		//lint -restore

		assert( TTV_SUCCEEDED(returning) );

		return returning;
	}


	TTV_WebcamFormat MediaTypeToWebcamFormat(AM_MEDIA_TYPE* const mediaType)
	{
		TTV_WebcamFormat mediaFormat = TTV_WEBCAM_FORMAT_UNKNOWN; // Any ol' dummy value will be fine

		// I would have used a switch here. But you cannot switch over non-integral types. So pretend this is a big switch.
		//if (mediaType->subtype == MEDIASUBTYPE_H264
			//|| MediaType->subtype == MEDIASUBTYPE_h246
			//|| MediaType->subtype == MEDIASUBTYPE_X264
			//|| MediaType->subtype == MEDIASUBTYPE_x264
			//|| MediaType->subtype == MEDIASUBTYPE_AVC1 ) // H.264 without start codes
			//)
		//{
			//mediaFormat = WebcamFormats::H264;
		//}
		/*else*/ if (mediaType->subtype == MEDIASUBTYPE_RGB1)
		{
			mediaFormat = TTV_WEBCAM_FORMAT_RGB1;
		}
		else if (mediaType->subtype == MEDIASUBTYPE_RGB4)
		{
			mediaFormat = TTV_WEBCAM_FORMAT_RGB4;
		}
		else if (mediaType->subtype == MEDIASUBTYPE_RGB8)
		{
			mediaFormat = TTV_WEBCAM_FORMAT_RGB8;
		}
		else if (mediaType->subtype == MEDIASUBTYPE_RGB555)
		{
			mediaFormat = TTV_WEBCAM_FORMAT_RGB555;
		}
		else if (mediaType->subtype == MEDIASUBTYPE_RGB565)
		{
			mediaFormat = TTV_WEBCAM_FORMAT_RGB565;
		}
		else if (mediaType->subtype == MEDIASUBTYPE_RGB24)
		{
			mediaFormat = TTV_WEBCAM_FORMAT_RGB24;
		}
		else if (mediaType->subtype == MEDIASUBTYPE_RGB32)
		{
			mediaFormat = TTV_WEBCAM_FORMAT_XRGB32;
		}
		else if (mediaType->subtype == MEDIASUBTYPE_ARGB1555)
		{
			mediaFormat = TTV_WEBCAM_FORMAT_ARGB1555;
		}
		else if (mediaType->subtype == MEDIASUBTYPE_ARGB32)
		{
			mediaFormat = TTV_WEBCAM_FORMAT_ARGB32;
		}
		else if (mediaType->subtype == MEDIASUBTYPE_ARGB4444)
		{
			mediaFormat = TTV_WEBCAM_FORMAT_ARGB4444;
		}
		else if (mediaType->subtype == MEDIASUBTYPE_A2R10G10B10)
		{
			// I know DirectShow calls it "ARGB" but it is actually in BGRA order
			mediaFormat = TTV_WEBCAM_FORMAT_B10G10R10A2;
		}
		else if (mediaType->subtype == MEDIASUBTYPE_A2B10G10R10)
		{
			// I know DirectShow calls it "ABGR" but it is actually in RGBA order
			mediaFormat = TTV_WEBCAM_FORMAT_R10G10B10A2;
		}
		else if (mediaType->subtype == MEDIASUBTYPE_AYUV)
		{
			mediaFormat = TTV_WEBCAM_FORMAT_AYUV;
		}
		else if (mediaType->subtype == MEDIASUBTYPE_YUY2)
		{
			mediaFormat = TTV_WEBCAM_FORMAT_YUY2;
		}
		else if (mediaType->subtype == MEDIASUBTYPE_UYVY)
		{
			mediaFormat = TTV_WEBCAM_FORMAT_UYVY;
		}
		else if (mediaType->subtype == MEDIASUBTYPE_IMC1)
		{
			mediaFormat = TTV_WEBCAM_FORMAT_IMC1;
		}
		else if (mediaType->subtype == MEDIASUBTYPE_IMC2)
		{
			mediaFormat = TTV_WEBCAM_FORMAT_IMC2;
		}
		else if (mediaType->subtype == MEDIASUBTYPE_IMC3)
		{
			mediaFormat = TTV_WEBCAM_FORMAT_IMC3;
		}
		else if (mediaType->subtype == MEDIASUBTYPE_IMC4)
		{
			mediaFormat = TTV_WEBCAM_FORMAT_IMC4;
		}
		else if (mediaType->subtype == MEDIASUBTYPE_YV12)
		{
			mediaFormat = TTV_WEBCAM_FORMAT_YV12;
		}
		else if (mediaType->subtype == MEDIASUBTYPE_NV12)
		{
			mediaFormat = TTV_WEBCAM_FORMAT_NV12;
		}

		return mediaFormat;
	}

	
	bool IsImageTopToBottom(const VIDEOINFOHEADER* const header, const TTV_WebcamFormat mediaFormat)
	{
		bool isTopToBottom = false;
	
		switch (mediaFormat)
		{
			// YUV images are always top-to-bottom, regardless of biHeight
			case TTV_WEBCAM_FORMAT_AYUV:
			case TTV_WEBCAM_FORMAT_YUY2:
			case TTV_WEBCAM_FORMAT_UYVY:
			case TTV_WEBCAM_FORMAT_IMC1:
			case TTV_WEBCAM_FORMAT_IMC2:
			case TTV_WEBCAM_FORMAT_IMC3:
			case TTV_WEBCAM_FORMAT_IMC4:
			case TTV_WEBCAM_FORMAT_YV12:
			case TTV_WEBCAM_FORMAT_NV12:
				isTopToBottom = true;
				break;

			// The normal image row order for RGB images is bottom-to-top, that is last row goes first. The top-to-bottom format 
			// also exists and goes with negative value in biHeight field of underlying media type.
			default:
				isTopToBottom = header->bmiHeader.biHeight < 0;
				break;
		}

		return isTopToBottom;
	}

	
	TTV_ErrorCode CreateCaptureGraphBuilder(std::unique_ptr<ICaptureGraphBuilder2, ttv::COMObjectDeleter<ICaptureGraphBuilder2>>& captureGraphBuilder, const std::function<TTV_ErrorCode()>& nextStep)
	{
		auto returning = TTV_EC_SUCCESS;

		// Initialize our stuff
		ICaptureGraphBuilder2* temporaryCaptureGraphBuilder = nullptr;
		returning = CreateCOMObject(CLSID_CaptureGraphBuilder2, IID_ICaptureGraphBuilder2, reinterpret_cast<VOID**>(&temporaryCaptureGraphBuilder));
		captureGraphBuilder.reset(temporaryCaptureGraphBuilder);

		// Call the next step
		if (TTV_SUCCEEDED(returning))
		{
			returning = nextStep();
		}

		// Clean up our stuff
		if (TTV_FAILED(returning))
		{
			captureGraphBuilder.reset(nullptr);
		}

		return returning;
	}

	TTV_ErrorCode CreateGraph(std::unique_ptr<IGraphBuilder, ttv::COMObjectDeleter<IGraphBuilder>>& graph, std::unique_ptr<ICaptureGraphBuilder2, ttv::COMObjectDeleter<ICaptureGraphBuilder2>>& captureGraphBuilder, const std::function<TTV_ErrorCode()>& nextStep)
	{
		auto returning = TTV_EC_SUCCESS;

		// Initialize our stuff
		IGraphBuilder* temporaryGraph = nullptr;
		returning = CreateCOMObject(CLSID_FilterGraph, IID_IGraphBuilder, reinterpret_cast<VOID**>(&temporaryGraph));
		graph.reset(temporaryGraph);
		if (graph != nullptr)
		{
			auto result = captureGraphBuilder->SetFiltergraph(graph.get());

			//lint -save
			//lint -e1924 C-style cast
			// This is coming from things like S_OK, which are not mine to control.
			switch (result)
			{
			case S_OK: // success
				break;
			case E_POINTER:
				// fallthrough
			case E_UNEXPECTED: // a filter graph has already been created
				returning = TTV_EC_WEBCAM_COULD_NOT_COMPLETE;
				break;
			default:
				returning = TTV_EC_WEBCAM_UNKNOWN_ERROR;
			}
			//lint -restore

			assert( TTV_SUCCEEDED(returning) );
		}

		// Call the next step
		if (TTV_SUCCEEDED(returning))
		{
			returning = nextStep();
		}

		// Clean up our stuff
		if (TTV_FAILED(returning))
		{
			graph.reset(nullptr);
		}

		return returning;
	}


	TTV_ErrorCode CreateMediaControl(std::unique_ptr<IGraphBuilder, ttv::COMObjectDeleter<IGraphBuilder>>& graph, std::unique_ptr<IMediaControl, ttv::COMObjectDeleter<IMediaControl>>& mediaControl, const std::function<TTV_ErrorCode()>& nextStep)
	{
		auto returning = TTV_EC_SUCCESS;

		// Initialize our stuff
		IMediaControl* temporaryMediaControl = nullptr;
		returning = QueryCOMInterface(graph.get(), IID_IMediaControl, reinterpret_cast<void**>(&temporaryMediaControl));
		mediaControl.reset(temporaryMediaControl);
	
		// Call the next step
		if (TTV_SUCCEEDED(returning))
		{
			returning = nextStep();
		}

		// Clean up our stuff
		if (TTV_FAILED(returning))
		{
			mediaControl.reset(nullptr);
		}

		return returning;
	}


	TTV_ErrorCode CreateMediaEvent(std::unique_ptr<IGraphBuilder, ttv::COMObjectDeleter<IGraphBuilder>>& graph, std::unique_ptr<IMediaEventEx, ttv::COMObjectDeleter<IMediaEventEx>>& event, const std::function<TTV_ErrorCode()>& nextStep)
	{
		auto returning = TTV_EC_SUCCESS;

		// Initialize our stuff
		IMediaEventEx* temporaryMediaEvent = nullptr;
		returning = QueryCOMInterface(graph.get(), IID_IMediaEventEx, reinterpret_cast<void**>(&temporaryMediaEvent));
		event.reset(temporaryMediaEvent);
	
		// Call the next step
		if (TTV_SUCCEEDED(returning))
		{
			returning = nextStep();
		}

		// Clean up our stuff
		if (TTV_FAILED(returning))
		{
			event.reset(nullptr);
		}

		return returning;
	}


	TTV_ErrorCode AddWebcamToGraph(std::unique_ptr<IGraphBuilder, ttv::COMObjectDeleter<IGraphBuilder>>& graph, std::shared_ptr<IBaseFilter>& source, const std::function<TTV_ErrorCode()>& nextStep)
	{
		auto returning = TTV_EC_SUCCESS;

		// Initialize our stuff
		returning = AddFilterToGraph(graph, source.get(), std::move(std::wstring(L"Webcam Capture")));

	
		// Call the next step
		if (TTV_SUCCEEDED(returning))
		{
			returning = nextStep();
		}

		// Clean up our stuff
		if (TTV_FAILED(returning))
		{
			// Nothing to clean up for this step
		}

		return returning;
	}


	TTV_ErrorCode CreateStreamConfig(std::unique_ptr<ICaptureGraphBuilder2, ttv::COMObjectDeleter<ICaptureGraphBuilder2>>& captureGraphBuilder, std::shared_ptr<IBaseFilter>& source, std::unique_ptr<IAMStreamConfig, ttv::COMObjectDeleter<IAMStreamConfig>>& streamConfig, const std::function<TTV_ErrorCode()>& nextStep)
	{
		auto returning = TTV_EC_SUCCESS;

		// Initialize our stuff
		IAMStreamConfig* temporaryStreamConfig = nullptr;
		auto result = captureGraphBuilder->FindInterface(NULL, &MEDIATYPE_Video, source.get(), IID_IAMStreamConfig, reinterpret_cast<void**>(&temporaryStreamConfig));
		streamConfig.reset(temporaryStreamConfig);

		//lint -save
		//lint -e1924 C-style cast
		// This is coming from things like S_OK, which are not mine to control.
		switch (result)
		{
		case S_OK: // success
			break;
		case E_FAIL:
			// fall through
		case E_NOINTERFACE: // The interface was not found
			// fall through
		case E_POINTER:
			// should not happen
			returning = TTV_EC_WEBCAM_COULD_NOT_COMPLETE;
			break;
		default:
			returning = TTV_EC_WEBCAM_UNKNOWN_ERROR;
		}
		//lint -restore

		assert( TTV_SUCCEEDED(returning) );
	
		// Call the next step
		if (TTV_SUCCEEDED(returning))
		{
			returning = nextStep();
		}

		// Clean up our stuff
		if (TTV_FAILED(returning))
		{
			streamConfig.reset(nullptr);
		}

		return returning;
	}


	TTV_ErrorCode SetCapability(AM_MEDIA_TYPE* const mediaType, std::unique_ptr<IAMStreamConfig, ttv::COMObjectDeleter<IAMStreamConfig>>& streamConfig, const std::function<TTV_ErrorCode()>& nextStep)
	{
		auto returning = TTV_EC_SUCCESS;

		// Initialize our stuff
		auto result = streamConfig->SetFormat(mediaType);

		assert(TTV_SUCCEEDED(returning));

		//lint -save
		//lint -e1924 C-style cast
		// This is coming from things like S_OK, which are not mine to control.
		switch (result)
		{
		case S_OK: // success
			break;
		case E_OUTOFMEMORY:
			returning = TTV_EC_WEBCAM_OUT_OF_MEMORY;
			break;
		case E_POINTER:
			// fall through
		case VFW_E_INVALIDMEDIATYPE:
			// fall through
		case VFW_E_NOT_CONNECTED: // The input pin is not connected
			// fall through
		case VFW_E_NOT_STOPPED:
			// fall through
		case VFW_E_WRONG_STATE:
			// should not happen
			returning = TTV_EC_WEBCAM_COULD_NOT_COMPLETE;
			break;
		default:
			returning = TTV_EC_WEBCAM_UNKNOWN_ERROR;
		}
		//lint -restore

	
		// Call the next step
		if (TTV_SUCCEEDED(returning))
		{
			returning = nextStep();
		}

		// Clean up our stuff
		if (TTV_FAILED(returning))
		{
			// Nothing to clean up for this step
		}

		return returning;
	}


	TTV_ErrorCode AddFrameGrabberToGraph(std::unique_ptr<IBaseFilter, ttv::COMObjectDeleter<IBaseFilter>>& grabberFilter, std::unique_ptr<IGraphBuilder, ttv::COMObjectDeleter<IGraphBuilder>>& graph, std::shared_ptr<IBaseFilter>& source, const AM_MEDIA_TYPE* const mediaType, std::unique_ptr<ISampleGrabber, ttv::COMObjectDeleter<ISampleGrabber>>& sampleGrabber, const std::function<TTV_ErrorCode()>& nextStep)
	{
		auto returning = TTV_EC_SUCCESS;

		// Initialize our stuff
		IBaseFilter* temporaryGrabberFilter = nullptr;
		returning = CreateCOMObject(CLSID_SampleGrabber, IID_IBaseFilter, reinterpret_cast<VOID**>(&temporaryGrabberFilter));
		grabberFilter.reset(temporaryGrabberFilter);
		if (TTV_SUCCEEDED(returning))
		{
			returning = AddFilterToGraph(graph, grabberFilter.get(), std::move(std::wstring(L"Sample Grabber")));
			if (TTV_SUCCEEDED(returning))
			{
				returning = ConnectFirstPins(source.get(), PINDIR_OUTPUT, grabberFilter.get(), PINDIR_INPUT, graph.get(), mediaType);
				if (TTV_SUCCEEDED(returning))
				{
					ISampleGrabber* temporarySampleGrabber = nullptr;
					returning = QueryCOMInterface(grabberFilter.get(), IID_ISampleGrabber, reinterpret_cast<void**>(&temporarySampleGrabber));
					sampleGrabber.reset(temporarySampleGrabber);
				}
			}
		}

	
		// Call the next step
		if (TTV_SUCCEEDED(returning))
		{
			returning = nextStep();
		}

		// Clean up our stuff
		if (TTV_FAILED(returning))
		{
			sampleGrabber.reset(nullptr);
		}

		return returning;
	}


	TTV_ErrorCode AddNullRendererToGraph(std::unique_ptr<IGraphBuilder, ttv::COMObjectDeleter<IGraphBuilder>>& graph, std::unique_ptr<IBaseFilter, ttv::COMObjectDeleter<IBaseFilter>>& grabberFilter, const AM_MEDIA_TYPE* const mediaType, const std::function<TTV_ErrorCode()>& nextStep)
	{
		auto returning = TTV_EC_SUCCESS;

		// Initialize our stuff
		IBaseFilter* temporaryNullRendererFilter = nullptr;
		returning = CreateCOMObject(CLSID_NullRenderer, IID_IBaseFilter, reinterpret_cast<VOID**>(&temporaryNullRendererFilter));
		std::unique_ptr<IBaseFilter, ttv::COMObjectDeleter<IBaseFilter>> nullRendererFilter(temporaryNullRendererFilter);

		if (TTV_SUCCEEDED(returning))
		{
			returning = AddFilterToGraph(graph, nullRendererFilter.get(), std::move(std::wstring(L"Null Renderer")));
			if (TTV_SUCCEEDED(returning))
			{
				returning = ConnectFirstPins(grabberFilter.get(), PINDIR_OUTPUT, nullRendererFilter.get(), PINDIR_INPUT, graph.get(), mediaType);
			}
		}


		// Call the next step
		if (TTV_SUCCEEDED(returning))
		{
			returning = nextStep();
		}

		// Clean up our stuff
		if (TTV_FAILED(returning))
		{
			// Nothing to clean up this step
		}

		return returning;
	}


	TTV_ErrorCode RunGraph(std::unique_ptr<IMediaControl, ttv::COMObjectDeleter<IMediaControl>>& mediaControl, const std::function<TTV_ErrorCode()>& nextStep)
	{
		auto returning = TTV_EC_SUCCESS;

		// Initialize our stuff
		auto result = mediaControl->Run();
		//lint -save
		//lint -e1924 C-style cast
		// This is coming from things like S_OK, which are not mine to control.
		switch (result)
		{
		case S_OK: // success
			// fallthrough
		case S_FALSE: // Some filters have not yet transitioned into running. But otherwise, things are fine
			break;
		case static_cast<HRESULT>(ERROR_DEVICE_NOT_CONNECTED): // The device is not connected.
			returning = TTV_EC_WEBCAM_UNKNOWN_ERROR;
			break;
		default:
			returning = TTV_EC_WEBCAM_UNKNOWN_ERROR;
			// TODO: a value of 0x8007001f here means "A device attached to the system is not functioning."
			// It happens when I switch from cam 1 to cam 2, then back to cam 1, fails on that second transition
			// http://social.msdn.microsoft.com/Forums/en-US/windowsdirectshowdevelopment/thread/0278445c-c381-46f6-920b-b285b2a1a9ba/
			// That link seems to indicate that it may be a USB bandwidth issue. So maybe just spin on shutdown to allow the shutdown to complete all the way?
		}
		//lint -restore
																					

		// Call the next step
		if (TTV_SUCCEEDED(returning))
		{
			returning = nextStep();
		}

		// Clean up our stuff
		if (TTV_FAILED(returning))
		{
			// Nothing to clean up this step
		}

		return returning;
	}


	TTV_ErrorCode InitializeCaptureFrames(const TTV_WebCamDeviceCapability& capability, std::array<TTV_WebCamFrame, 2>& frames, volatile int& captureFrame, unsigned int deviceIndex, const std::function<TTV_ErrorCode()>& nextStep)
	{
		auto returning = TTV_EC_SUCCESS;

		// Initialize our stuff
		size_t bufferSizeInBytes = 0;
		returning = GetBufferSize(capability, bufferSizeInBytes);

		if (TTV_SUCCEEDED(returning))
		{
			for (size_t i = 0; i < frames.size(); ++i)
			{
				frames[i].deviceIndex = deviceIndex;
				frames[i].bufferSize = static_cast<unsigned int>(bufferSizeInBytes);
				frames[i].capability = capability;
				frames[i].imageBuffer = new uint8_t[bufferSizeInBytes];

				memset(frames[i].imageBuffer, 0, bufferSizeInBytes);
			}
		}

		// set the initial frame to capture to
		captureFrame = 0;
																					

		// Call the next step
		if (TTV_SUCCEEDED(returning))
		{
			returning = nextStep();
		}

		// Clean up our stuff
		if (TTV_FAILED(returning))
		{
			for (size_t i = 0; i < frames.size(); ++i)
			{
				delete[] frames[i].imageBuffer;
			}
		}

		return returning;
	}


	TTV_ErrorCode StartCallbacks(ttv::cam::SampleGrabberCallback& callback, std::unique_ptr<ISampleGrabber, ttv::COMObjectDeleter<ISampleGrabber>>& sampleGrabber, const std::shared_ptr<ttv::cam::DirectShowVideoCaptureDevice>& that)
	{
		auto returning = TTV_EC_SUCCESS;

		// Initialize our stuff
		callback.Initialize(that);
		auto result = sampleGrabber->SetCallback(&callback, 1 /* Call buffer callback instead of sample callback */);
		//lint -save
		//lint -e1924 C-style cast
		// This is caused by the FAILED() macro, which is beyond my control.
		if (FAILED(result))
		//lint -restore
		{
			returning = TTV_EC_WEBCAM_COULD_NOT_COMPLETE;
		}

		// There is no next step

		// There is no clean up step for this

		return returning;
	}

} // unnamed namespace


ttv::cam::DirectShowVideoCaptureDevice::DirectShowVideoCaptureDevice(ttv::cam::VideoCaptureSystem* parentSystem, std::shared_ptr<IBaseFilter> source, const std::string& name, const std::string& devicePath)
:	ttv::cam::VideoCaptureDevice(parentSystem)
,	std::enable_shared_from_this<ttv::cam::DirectShowVideoCaptureDevice>()
,	mSource(source)
,	mCaptureGraphBuilder(nullptr)
,	mGraph(nullptr)
,	mMediaControl(nullptr)
,	mEvent(nullptr)
,	mSampleGrabber(nullptr)
,	mStreamConfig(nullptr)
,	mCallback()
{
	FillDeviceInfo(mClientDeviceInfo, name, devicePath);

	// Leave the capabilities empty until init is called
	mClientDeviceInfo.capabilityList.count = 0;
}


ttv::cam::DirectShowVideoCaptureDevice::~DirectShowVideoCaptureDevice()
{
	assert(mMediaTypes.size() == 0);

	// It is possible that another thread called the callback while this is about to be destroyed.
	// The docs - http://msdn.microsoft.com/en-us/library/windows/desktop/dd376992(v=vs.85).aspx - give no
	// guarantee that this will block until the callback will not be called again.

	// Unfortunately, there is no way to signal that other thread that it is supposed to shut down.
	// Nor can we wait on it.
	// The best we can do is either:
	// 1.) attempt to shut down and then pause for maybe a frame or two's time, hoping that it has exited, or
	// 2.) hope that this function blocks until the callback is not going to be called again.

	// I am going to go with option 2.
	if (mSampleGrabber != nullptr)
	{
		(void)mSampleGrabber->SetCallback(NULL, 1 /* Call buffer callback instead of sample callback */);
	}

	if (mMediaControl != nullptr)
	{
		const HRESULT result = mMediaControl->Stop();
		switch (result)
		{
			//lint -save
			//lint -e1924 C-style cast
			// This is coming from S_OK, which are not mine to control.
		case S_OK: // success
			//lint -restore
			break;
		default:
			// Although an error occurred, because we are in a shut-down-type function, just ignore the error and continue to shut down.
			;
		}
	}
	
	mSource.reset();
}


TTV_ErrorCode ttv::cam::DirectShowVideoCaptureDevice::InitializeDevice(const ttv::cam::InitializeDeviceMessage* /*message*/)
{
	auto returning = TTV_EC_SUCCESS;

	auto result = CoInitializeEx(NULL, COINIT_APARTMENTTHREADED);
	//lint -save
	//lint -e1924 C-style cast
	// This is coming from things like S_OK, which are not mine to control.
	switch (result)
	{
	case S_OK:
		// Success
		// Fall through
	case S_FALSE:
		// Already initialized on this thread, but that is okay.
		break;
	case RPC_E_CHANGED_MODE:
		// A previous call used a different apartment model.
		returning = TTV_EC_WEBCAM_UNKNOWN_ERROR;
		break;
	case E_OUTOFMEMORY:
		returning = TTV_EC_WEBCAM_OUT_OF_MEMORY;
		break;
	case E_INVALIDARG:
		// Fall through
	case E_UNEXPECTED:
		// Fall through
	default:
		returning = TTV_EC_WEBCAM_UNKNOWN_ERROR;
	}
	//lint -restore

	if (TTV_SUCCEEDED(returning))
	{
		// Fill the supported media types
		returning = GetSupportedMediaParameters();
	}

	return returning;
}


TTV_ErrorCode ttv::cam::DirectShowVideoCaptureDevice::ShutdownDevice(const ttv::cam::ShutdownDeviceMessage* /*message*/)
{
	for (size_t i=0; i<mMediaTypes.size(); ++i)
	{
#pragma warning(suppress: 6001) // The memory is initialized or else it wouldn't be pushed onto the vector
		CoTaskMemFree(mMediaTypes[i]);
	}

	mMediaTypes.clear();
	mSource.reset();

	CoUninitialize();

	return TTV_EC_SUCCESS;
}


TTV_ErrorCode ttv::cam::DirectShowVideoCaptureDevice::StartDevice(const ttv::cam::StartDeviceMessage* message)
{
	auto returning = TTV_EC_SUCCESS;

	if (message == nullptr)
	{
		returning = TTV_EC_WEBCAM_INVALID_PARAMETER;
	}
	else
	{
		if (mStatus != TTV_WEBCAM_DEVICE_STOPPED)
		{
			returning = TTV_EC_WEBCAM_DEVICE_ALREADY_STARTED;
		}
		else
		{
			int listIndex = FindCapability(message->capabilityIndex);

			if (listIndex < 0)
			{
				returning = TTV_EC_WEBCAM_INVALID_PARAMETER;
			}
			else
			{
				const TTV_WebCamDeviceCapability& capability = mClientDeviceInfo.capabilityList.list[listIndex]; //-V108   Incorrect index type: ___[not a memsize-type]. Use memsize type instead.

				if (mCaptureGraphBuilder != nullptr)
				{
					returning = TTV_EC_WEBCAM_DEVICE_ALREADY_STARTED;
					assert(false);
				}
				else
				{
					AM_MEDIA_TYPE* const mediaType = mMediaTypes[GetParentCapability(capability).capabilityIndex]; //-V108   Incorrect index type: ____[not a memsize-type]. Use memsize type instead.
					std::unique_ptr<IBaseFilter, ttv::COMObjectDeleter<IBaseFilter>> grabberFilter;

					// These are the initializations that need to take place.
					// Please note: They are in reverse order.
					// The first one listed is the last one executed.
					// Notice, it passes the std::function to the next line as the final parameter.
					// It is sort of building up a stack of "This, then this, then that..."
					std::function<TTV_ErrorCode()> startCallbacks = std::bind(StartCallbacks, std::ref(mCallback), std::ref(mSampleGrabber), std::cref(shared_from_this()));
					std::function<TTV_ErrorCode()> initializeCaptureFrames = std::bind(InitializeCaptureFrames, std::ref(capability), std::ref(mFrames), std::ref(mCaptureFrame), mClientDeviceInfo.deviceIndex, startCallbacks);
					std::function<TTV_ErrorCode()> runGraph = std::bind(RunGraph, std::ref(mMediaControl), initializeCaptureFrames);
					std::function<TTV_ErrorCode()> addNullRendererToGraph = std::bind(AddNullRendererToGraph, std::ref(mGraph), std::ref(grabberFilter), std::cref(mediaType), runGraph);
					std::function<TTV_ErrorCode()> addFrameGrabberToGraph = std::bind(AddFrameGrabberToGraph, std::ref(grabberFilter), std::ref(mGraph), std::ref(mSource), mediaType, std::ref(mSampleGrabber), addNullRendererToGraph);
					std::function<TTV_ErrorCode()> setCapability = std::bind(SetCapability, std::cref(mediaType), std::ref(mStreamConfig), addFrameGrabberToGraph);
					std::function<TTV_ErrorCode()> createStreamConfig = std::bind(CreateStreamConfig, std::ref(mCaptureGraphBuilder), std::ref(mSource), std::ref(mStreamConfig), setCapability);
					std::function<TTV_ErrorCode()> addWebcamToGraph = std::bind(AddWebcamToGraph, std::ref(mGraph), std::ref(mSource), createStreamConfig);
					std::function<TTV_ErrorCode()> createMediaEvent = std::bind(CreateMediaEvent, std::ref(mGraph), std::ref(mEvent), addWebcamToGraph);
					std::function<TTV_ErrorCode()> createMediaControl = std::bind(CreateMediaControl, std::ref(mGraph), std::ref(mMediaControl), createMediaEvent);
					std::function<TTV_ErrorCode()> createGraph = std::bind(CreateGraph, std::ref(mGraph), std::ref(mCaptureGraphBuilder), createMediaControl);
					returning = CreateCaptureGraphBuilder(mCaptureGraphBuilder, createGraph);
				}
			}
		}

		if (TTV_SUCCEEDED(returning))
		{
			mStatus = TTV_WEBCAM_DEVICE_STARTED;

			int listIndex = FindCapability(message->capabilityIndex);
			assert(listIndex >= 0);

			const TTV_WebCamDeviceCapability& capability = mClientDeviceInfo.capabilityList.list[listIndex]; //-V108   Incorrect index type: ___[not a memsize-type]. Use memsize type instead.

			// notify the client
			SendEventToClient( std::make_shared<ttv::cam::DeviceStatusClientMessage>(mClientDeviceInfo.deviceIndex, mStatus, capability, message->callback, message->userdata) );
		}
		else
		{
			mStatus = TTV_WEBCAM_DEVICE_STOPPED;

			// notify the client
			SendEventToClient( std::make_shared<ttv::cam::DeviceStatusClientMessage>(mClientDeviceInfo.deviceIndex, mStatus, message->capabilityIndex, returning, message->callback, message->userdata) );
		}
	}

	return returning;
}


TTV_ErrorCode ttv::cam::DirectShowVideoCaptureDevice::StopDevice(const ttv::cam::StopDeviceMessage* message)
{
	assert(mDeviceThreadRunning);

	if (mStatus != TTV_WEBCAM_DEVICE_STARTED)
	{
		return TTV_EC_WEBCAM_DEVICE_NOT_STARTED;
	}

	mStatus = TTV_WEBCAM_DEVICE_STOPPED;

	auto returning = TTV_EC_SUCCESS;

	if (mSampleGrabber != nullptr)
	{
		// It is possible that another thread called the callback while this is about to be destroyed.
		// The docs - http://msdn.microsoft.com/en-us/library/windows/desktop/dd376992(v=vs.85).aspx - give no
		// guarantee that this will block until the callback will not be called again.

		// Unfortunately, there is no way to signal that other thread that it is supposed to shut down.
		// Nor can we wait on it.
		// The best we can do is either:
		// 1.) attempt to shut down and then pause for maybe a frame or two's time, hoping that it has exited, or
		// 2.) hope that this function blocks until the callback is not going to be called again.

		// I am going to go with option 2.
		HRESULT result = mSampleGrabber->SetCallback(NULL, 1 /* Call buffer callback instead of sample callback */);
		//lint -save
		//lint -e1924   C-style cast
		// This is from the FAILED() macro, which is beyond my control.
		if (FAILED(result))
		//lint -restore
		{
			returning = TTV_EC_WEBCAM_COULD_NOT_COMPLETE;
		}

		if (mMediaControl != nullptr)
		{
			result = mMediaControl->Stop();
			switch (result)
			{
			//lint -save
			//lint -e1924   C-style cast
			// This is coming from S_OK, which is not mine to control.
			case S_OK: // success
				break;
			//lint -restore
			default:
				// Although an error occurred, because we are in a shut-down-type function, just flag the error and continue to shut down.
				if (TTV_SUCCEEDED(returning))
				{
					returning = TTV_EC_WEBCAM_COULD_NOT_COMPLETE;
				}
			}
		}
	}

	mMediaControl.reset(nullptr);
	mSampleGrabber.reset(nullptr);
	mEvent.reset(nullptr);
	mCaptureGraphBuilder.reset(nullptr);
	mGraph.reset(nullptr);
	mStreamConfig.reset(nullptr);

	// stop the client thread from receiving frame updates
	ClearFrames();

	// NOTE: The assumption which hasn't been 100% confirmed is that the sample grabber callback won't be called anymore and 
	// will be guaranteed to not be executing when mSampleGrabber->SetCallback(NULL) returns.  It seems to be the case though.

	// deallocate buffers
	(void)mMutex->Lock();
	for (size_t i = 0; i < mFrames.size(); ++i)
	{
		delete [] mFrames[i].imageBuffer;
		mFrames[i].imageBuffer = nullptr;
	}
	(void)mMutex->Unlock();

	// notify the client
	if (message != nullptr)
	{
		int listIndex = FindCapability(mFrames[0].capability.capabilityIndex);
		assert(listIndex >= 0);

		const TTV_WebCamDeviceCapability& capability = mClientDeviceInfo.capabilityList.list[listIndex];

		SendEventToClient(std::make_shared<ttv::cam::DeviceStatusClientMessage>(mClientDeviceInfo.deviceIndex, TTV_WEBCAM_DEVICE_STOPPED, capability, message->callback, message->userdata));
	}

	return returning;
}


TTV_ErrorCode ttv::cam::DirectShowVideoCaptureDevice::GetSupportedMediaParameters()
{
	auto returning = TTV_EC_SUCCESS;

	// find all the output pins for the device
	std::vector<IPin*> outputPins;
	returning = GetPins(mSource.get(), PINDIR_OUTPUT, outputPins);

	if (TTV_FAILED(returning))
	{
		return returning;
	}

	// examine all the pins until we find one that suits our needs
	for (auto currentPin = outputPins.begin(); currentPin != outputPins.end(); ++currentPin)
	{
		bool continueEnumeratingPins = true;

		IEnumMediaTypes* mediaTypes = nullptr;
		HRESULT result = (*currentPin)->EnumMediaTypes(&mediaTypes);

		//lint -save
		//lint -e1924 C-style cast
		// This is coming from things like S_OK, which are not mine to control.
		switch (result)
		{
		case S_OK: // success
			break;
		case E_OUTOFMEMORY:
			returning = TTV_EC_WEBCAM_OUT_OF_MEMORY;
			break;
		case E_POINTER:
			// fallthrough
		case VFW_E_NOT_CONNECTED:
		case VFW_E_ENUM_OUT_OF_SYNC:
			returning = TTV_EC_WEBCAM_COULD_NOT_COMPLETE;
			break;
		default:
			returning = TTV_EC_WEBCAM_UNKNOWN_ERROR;
		}
		//lint -restore

		assert(TTV_SUCCEEDED(returning));

		if (TTV_SUCCEEDED(returning) && mediaTypes != nullptr)
		{
			std::vector<TTV_WebCamDeviceCapability> temporaryMediaTypes;
			bool moreMediaTypesLeft = true;
			bool nonStillShotFound = false;

			// examine each media type the pin supports
			do
			{
				bool keepMediaType = false;
				AM_MEDIA_TYPE* mediaType = nullptr;
				result = mediaTypes->Next(1, &mediaType, NULL);

				//lint -save
				//lint -e1924 C-style cast
				// This is coming from things like S_OK, which are not mine to control.
				switch (result)
				{
				case S_OK:
					break;
				case S_FALSE: // No more media types
					moreMediaTypesLeft = false;
					break;
				case E_INVALIDARG:
					// fall through
				case E_POINTER:
					returning = TTV_EC_WEBCAM_COULD_NOT_COMPLETE;
					break;
				case VFW_E_ENUM_OUT_OF_SYNC: // The pin's state has changed and the enumerator is now invalid
					// should dump whatever data we have found so far and then call ->Reset
					returning = TTV_EC_WEBCAM_COULD_NOT_COMPLETE;
					break;
				default:
					returning = TTV_EC_WEBCAM_UNKNOWN_ERROR;
				}
				//lint -restore

				assert(TTV_SUCCEEDED(returning));

				if (TTV_SUCCEEDED(returning) && mediaType != nullptr)
				{
					if (mediaType->formattype == FORMAT_VideoInfo &&
						mediaType->majortype == MEDIATYPE_Video &&
						mediaType->cbFormat >= sizeof(VIDEOINFOHEADER))
					{
						TTV_WebcamFormat mediaFormat = MediaTypeToWebcamFormat(mediaType);

						//lint -save
						//lint -e826   Suspicious pointer-to-pointer conversion (area too small)
						// This is because a BYTE (mediaType->pbFormat's type) is smaller than a VIDEOINFOHEADER
						// However, we know this happens to be a VIDEOINFOHEADER. That is why we checked mediaType->formattype.
						// So we can feel safe knowing that the conversion is okay. It was intended.
						const auto videoInfoHeader = reinterpret_cast<const VIDEOINFOHEADER*>(mediaType->pbFormat);
						//lint -restore
						const float conversionFactor100nsTos = 1.0f / 10000000.0f;
						const float frameRate = 1.0f / (static_cast<float>(videoInfoHeader->AvgTimePerFrame) * conversionFactor100nsTos);
						auto targetFrameRate = static_cast<unsigned int>(frameRate + 0.5f/*to round*/);

						// found a valid media type
						if (mediaFormat != TTV_WEBCAM_FORMAT_UNKNOWN &&
							targetFrameRate > 1)
						{
							nonStillShotFound = true;

							TTV_WebCamDeviceCapability currentMediaParameters;
							currentMediaParameters.capabilityIndex = static_cast<unsigned int>(temporaryMediaTypes.size());
							currentMediaParameters.resolution.height = abs(videoInfoHeader->bmiHeader.biHeight);
							currentMediaParameters.resolution.width = videoInfoHeader->bmiHeader.biWidth;
							currentMediaParameters.frameRate = targetFrameRate;
							currentMediaParameters.format = mediaFormat;
							currentMediaParameters.isTopToBottom = IsImageTopToBottom(videoInfoHeader, mediaFormat);
							currentMediaParameters.isNative = true;
							temporaryMediaTypes.push_back(currentMediaParameters);

							keepMediaType = true;
							mMediaTypes.push_back(mediaType);
						}

						// ISampleGrabber does not support VIDEOINFOHEADER2
					}
				}

				// free the media type if not cached
				if (mediaType != nullptr && !keepMediaType)
				{
					//lint -save
					//lint -e10   Expecting a function
					// This is to do with function pointers, presumably. void* foo() vs. void (*foo)(). CoTaskMemFree is NOT expecting a function, as lint is claiming. May be a bug in lint (http://www.gimpel.com/Discussion.cfm?ThreadID=2987)
					(void)CoTaskMemFree(mediaType);
					//lint -restore
				}

			} while(moreMediaTypesLeft == true);

			// add support for the automatically converted non-native formats
			ExtendCapabilities(temporaryMediaTypes);
					
			// now that we're examines all the media types on the pin, use these capabilities if there is a valid framerate on any of them
			if (nonStillShotFound)
			{
				const size_t capabilityCount = temporaryMediaTypes.size();
				mClientDeviceInfo.capabilityList.count = static_cast<unsigned int>(capabilityCount);
				mClientDeviceInfo.capabilityList.list = new TTV_WebCamDeviceCapability[capabilityCount];

				for (size_t i = 0; i < capabilityCount; ++i)
				{
					mClientDeviceInfo.capabilityList.list[i] = temporaryMediaTypes[i];
				}
			}

			// we can only handle one video pin so take it
			if (nonStillShotFound)
			{
				continueEnumeratingPins = false;
			}
		}

		if (mediaTypes != nullptr)
		{
			(void)mediaTypes->Release();
		}

		(void)(*currentPin)->Release();

		if (!continueEnumeratingPins)
		{
			break;
		}
	}

	// check for valid device
	if (mClientDeviceInfo.capabilityList.count == 0)
	{
		returning = TTV_EC_WEBCAM_INVALID_CAPABILITY;
	}

	return returning;
}


void ttv::cam::DirectShowVideoCaptureDevice::Update()
{
	if (mEvent == nullptr)
	{
		return;
	}

	// The docs suggest that once awoken, the code pull all the events queued and then go to sleep after the queue is empty.
	bool moreEventsQueued = true;
	do
	{
		long code = 0;
		LONG_PTR param1 = NULL;
		LONG_PTR param2 = NULL;
		long waitTimeInMilliseconds = 0; // Do not wait for an event
		auto result = mEvent->GetEvent(&code, &param1, &param2, waitTimeInMilliseconds);
		//lint -save
		//lint -e1924 C-style cast
		// This is coming from things like S_OK, which are not mine to control.
		switch (result)
		{
		case S_OK: // success
			break;
		case E_ABORT: // timed out (or the event queue was empty?)
			moreEventsQueued = false;
			break;
		default:
			// TODO: Cannot return errors just yet
			//returning = TTV_EC_WEBCAM_UNKNOWNERROR;
			moreEventsQueued = false;
		}
		//lint -restore

		if (moreEventsQueued == true)
		{
			switch (code)
			{
			// We only really care about events which prevent us from capturing

			case EC_DEVICE_LOST:
				// Fall through
			case EC_STREAM_ERROR_STOPPED: // The stream has stopped because of an error
				// Fall through
			
			case EC_ERRORABORT:
				// Fall through
			case EC_ERRORABORTEX:
				// TODO: Cannot return errors just yet
				// So ignore the return value of Shutdown();
				(void)Shutdown();
				break;

			default:
				// Do nothing
				;
			}

			(void)mEvent->FreeEventParams(code, param1, param2);
		}
	} while (moreEventsQueued);
}
