/********************************************************************************************
* Twitch Broadcasting SDK
*
* This software is supplied under the terms of a license agreement with Twitch Interactive, Inc. and
* may not be copied or used except in accordance with the terms of that agreement
* Copyright (c) 2012-2016 Twitch Interactive, Inc.
*********************************************************************************************/

#include "internal/pch.h"
#include "internal/webcam/win32/directshowvideocapturesystem.h"
#include "internal/webcam/win32/directshowvideocapturedevice.h"
#include "twitchcore/systemclock.h"

#include <Dbt.h>
#include <mfidl.h>


namespace
{
	bool ConvertString(std::wstring source, utf8char* buffer, size_t bufferLen);
} // unnamed namespace


namespace
{

	bool ConvertString(std::wstring source, utf8char* buffer, size_t bufferLen)
	{
		bool returning = false;

		std::string utf8str(source.begin(), source.end());
		auto len = utf8str.length();
		if (len >= bufferLen)
		{
			// Do NOT just copy in a portion.
			buffer[0] = '\0';
			assert(false);
		}
		else
		{
			memcpy(buffer, utf8str.c_str(), len);
			buffer[len] = 0;
			returning = true;
		}

		return returning;
	}

} // unnamed namespace

ttv::cam::DirectShowVideoCaptureSystem::DirectShowVideoCaptureSystem()
:	mLastDeviceChangeCheckTime(0)
{
}


ttv::cam::DirectShowVideoCaptureSystem::~DirectShowVideoCaptureSystem()
{
	//lint -save
	//lint -e1540   Pointer member ____ neither freed nor zeroed by destructor
	// This is because the real destruction happens inside the if.
}
//lint -restore


TTV_ErrorCode ttv::cam::DirectShowVideoCaptureSystem::InitializeSystem(const ttv::cam::InitializeSystemMessage* /*message*/)
{
	auto returning = TTV_EC_SUCCESS;

	auto result = CoInitializeEx(NULL, COINIT_APARTMENTTHREADED);
	//lint -save
	//lint -e1924 C-style cast
	// This is coming from things like S_OK, which are not mine to control.
	switch (result)
	{
	case S_OK:
		// Success
		// Fall through
	case S_FALSE:
		// Already initialized on this thread, but that is okay.
		break;
	case RPC_E_CHANGED_MODE:
		// A previous call used a different apartment model.
		returning = TTV_EC_WEBCAM_UNKNOWN_ERROR;
		break;
	case E_OUTOFMEMORY:
		returning = TTV_EC_WEBCAM_OUT_OF_MEMORY;
		break;
	case E_INVALIDARG:
		// Fall through
	case E_UNEXPECTED:
		// Fall through
	default:
		returning = TTV_EC_WEBCAM_UNKNOWN_ERROR;
	}
	//lint -restore

	returning = CheckForDeviceChanges();

	return returning;
}


TTV_ErrorCode ttv::cam::DirectShowVideoCaptureSystem::CheckForDeviceChanges()
{
	// TODO: this doesn't take into account a device which is unplugged and plugged back in quickly
	// TODO: this should be changed to an event-driven method

	// find the current devices
	std::vector<DirectShowDeviceInfo> current;
	(void)FindCurrentDevices(current);

	// create a copy of the previous devices
	std::vector< std::shared_ptr<ttv::cam::VideoCaptureDevice> > previous = mSystemDevices;

	// remove devices from the lists which are in the both lists
	for (auto c = current.begin(); c != current.end(); )
	{
		DirectShowDeviceInfo& cinfo = *c;
		bool found = false;

		// see if it's already in the list
		for (auto p = previous.begin(); p != previous.end(); ++p)
		{
			if ( strcmp(cinfo.uniqueId, (*p)->GetClientDeviceInfo().uniqueId) == 0 )
			{
				(void)previous.erase(p);
				found = true;
				break;
			}
		}

		if (found)
		{
			c = current.erase(c);
		}
		else
		{
			++c;
		}
	}

	// process removes
	for (auto p = previous.begin(); p != previous.end(); ++p)
	{
		std::shared_ptr<ttv::cam::VideoCaptureDevice> device = *p;
		(void)device->Shutdown();
	}

	// process adds
	for (auto c = current.begin(); c != current.end(); ++c)
	{
		DirectShowDeviceInfo& cinfo = *c;
		
		// create the device instance
		auto device = std::make_shared<ttv::cam::DirectShowVideoCaptureDevice>(this, cinfo.camera, cinfo.name, cinfo.uniqueId);
		mSystemDevices.push_back(device);

		// initialize the device
		(void)device->Initialize(mNextDeviceIndex);
		mNextDeviceIndex++;
	}

	return TTV_EC_SUCCESS;
}


TTV_ErrorCode ttv::cam::DirectShowVideoCaptureSystem::FindCurrentDevices(std::vector<DirectShowDeviceInfo>& list)
{
	auto returning = TTV_EC_SUCCESS;

	ICreateDevEnum* devices = nullptr;

	// TODO: do we want to try and cache the device enumerator?

	// create the device enumerator
	HRESULT result = CoCreateInstance(CLSID_SystemDeviceEnum, NULL, static_cast<DWORD>(CLSCTX_INPROC_SERVER), IID_ICreateDevEnum, reinterpret_cast<void**>(&devices));
	switch (result)
	{
	case S_OK: // succes
		break;
	case REGDB_E_CLASSNOTREG: // A class matching the specified class is not registered. Or the server types in the registry are corrupt.
		// fall through
	case E_NOINTERFACE: // The specified class does not implement the requested interface.
		returning = TTV_EC_WEBCAM_NO_PLATFORM_SUPPORT;
		break;
	case CLASS_E_NOAGGREGATION:
		// fall through
	case E_POINTER:
		returning = TTV_EC_WEBCAM_COULD_NOT_COMPLETE;
		break;
	default:
		returning = TTV_EC_WEBCAM_UNKNOWN_ERROR;
	}

	if (TTV_SUCCEEDED(returning) && devices != nullptr)
	{
		IEnumMoniker* enumerator;
		result = devices->CreateClassEnumerator(CLSID_VideoInputDeviceCategory, &enumerator, 0);

		switch (result)
		{
		case S_OK: // success
			// fall through
		case S_FALSE:
			// If enumerator is also NULL when S_FALSE was returned then the category specified does not exist or is empty.
			if (enumerator == nullptr)
			{
				returning = TTV_EC_WEBCAM_COULD_NOT_COMPLETE;
			}
			break;
		case E_OUTOFMEMORY:
			returning = TTV_EC_WEBCAM_OUT_OF_MEMORY;
			break;
		case E_POINTER: // NULL pointer argument.
			returning = TTV_EC_WEBCAM_COULD_NOT_COMPLETE;
			break;
		default:
			returning = TTV_EC_WEBCAM_UNKNOWN_ERROR;
		}

		if (TTV_SUCCEEDED(returning) && enumerator != nullptr) // If the category didn't exist or was empty, the enumerator pointer will be null
		{
			bool moreDevicesLeft = true;

			do
			{
				IMoniker* moniker = nullptr;
				result = enumerator->Next(1, &moniker, NULL);
				switch (result)
				{
				case S_OK:
					break;
				case S_FALSE: // No more items in the enumerator
					moreDevicesLeft = false;
					break;
				default:
					returning = TTV_EC_WEBCAM_UNKNOWN_ERROR;
				}

				if (TTV_SUCCEEDED(returning) && moniker != nullptr)
				{
					IBaseFilter* camera = nullptr;
#pragma warning(suppress: 6387) // We have not created any binding contexts intentionally
					result = moniker->BindToObject(NULL, NULL, IID_IBaseFilter, reinterpret_cast<void**>(&camera));
				
					// a reference to ensure Release() is called if not a valid device
					std::shared_ptr<IBaseFilter> cameraReference(camera, ttv::COMObjectDeleter<IBaseFilter>());

					switch (result)
					{
					case S_OK: // succeeded
						break;
					case E_OUTOFMEMORY:
						returning = TTV_EC_WEBCAM_OUT_OF_MEMORY;
						break;
					case MK_E_NOOBJECT: // The object identified by this moniker could not be found
						// fall through
					case MK_E_EXCEEDEDDEADLINE: // Timed out
						// fall through
					case MK_E_CONNECTMANUALLY: // This means the user must do something like input a password or mount a removable disk or some such.
						// Going to ignore this for now.
						// fall through
					case MK_E_INTERMEDIATEINTERFACENOTSUPPORTED: // An intermediate object was found but it did not support an interface required to complete the bind.
						// fall through
					case STG_E_ACCESSDENIED:
						returning = TTV_EC_WEBCAM_COULD_NOT_COMPLETE;
						break;
					case E_UNEXPECTED:
						// fall through
					default:
						returning = TTV_EC_WEBCAM_UNKNOWN_ERROR;
					}

					// Get the device's name
					IPropertyBag* propertyBag = nullptr;

#pragma warning(suppress: 6387) // We have not created any binding contexts intentionally
					result = moniker->BindToStorage(0, 0, IID_IPropertyBag, reinterpret_cast<void**>(&propertyBag));
					
					if (FAILED(result))
					{
						// Expected error codes:
						// MK_E_NOSTORAGE - This object does not have its own storage
						// MK_E_CONNECTMANUALLY - Could not connect (possibly because the network device could not connect)
						// MK_E_INTERMEDIATEINTERFACENOTSUPPORTED - The intermediate object cannot bind
						// MK_E_EXCEEDEDDEADLINE - Timed out
						// STG_E_ACCESSDENIED
						returning = TTV_EC_WEBCAM_COULD_NOT_COMPLETE;
					}

					if (TTV_SUCCEEDED(returning))
					{
						VARIANT name;
						VariantInit(&name);
						result = propertyBag->Read(L"FriendlyName", &name, 0);
						if (FAILED(result))
						{
							// Expected error codes:
							// E_POINTER
							// E_INVALIDARG
							// E_FAIL - Unable to read that property
							returning = TTV_EC_WEBCAM_COULD_NOT_COMPLETE;
						}

						if (TTV_SUCCEEDED(returning))
						{
							VARIANT devicePath;
							VariantInit(&devicePath);
							result = propertyBag->Read(L"DevicePath", &devicePath, 0);

							if (FAILED(result))
							{
								// Expected error codes:
								// E_POINTER
								// E_INVALIDARG
								// E_FAIL - Unable to read that property
								
								// Rather than fail to create this device and report an error...
								// skip this device.
								// The reason is some devices, such as "VDP Source" and "XSplit capture" do not have unique identifiers.
								// But they also aren't really webcams.
							}
							else
							{
								std::wstring uniqueId = devicePath.bstrVal;
								std::wstring deviceName = name.bstrVal;

								// We wish to remove virtual devices.
								// Sometimes they have no device path. Other times they have an empty string as a device path.
								// So check if the device path is empty.
								if (uniqueId.length() != 0)
								{
									// create the device info
									DirectShowDeviceInfo info;
									camera->AddRef();
									info.camera = std::shared_ptr<IBaseFilter>(camera, ttv::COMObjectDeleter<IBaseFilter>());
									(void)ConvertString(deviceName, info.name, sizeof(info.name)); // TODO: I think we should copy in as much of the name as we can. Not just copy none if too large.
									(void)ConvertString(uniqueId, info.uniqueId, sizeof(info.uniqueId));
									list.push_back(info);
								}

								result = VariantClear(&devicePath);
								if (FAILED(result))
								{
									// Expected error codes:
									// DISP_E_ARRAYISLOCKED
									// DISP_E_BADVARTYPE
									// E_INVALIDARG
									returning = TTV_EC_WEBCAM_COULD_NOT_COMPLETE;
								}
							}

							result = VariantClear(&name);
							if (FAILED(result))
							{
								// Expected error codes:
								// DISP_E_ARRAYISLOCKED
								// DISP_E_BADVARTYPE
								// E_INVALIDARG
								returning = TTV_EC_WEBCAM_COULD_NOT_COMPLETE;
							}
						}
					}

					if (propertyBag != nullptr)
					{
						(void)propertyBag->Release();
					}
				}

				if (moniker != nullptr)
				{
					(void)moniker->Release();
				}

			} while (moreDevicesLeft);

			(void)enumerator->Release();
		}
	}

	if (devices != nullptr)
	{
		(void)devices->Release();
	}

	return returning;
}


TTV_ErrorCode ttv::cam::DirectShowVideoCaptureSystem::ShutdownSystem(const ttv::cam::ShutdownSystemMessage* /*message*/)
{
	CoUninitialize();

	return TTV_EC_SUCCESS;
}


void ttv::cam::DirectShowVideoCaptureSystem::Update()
{
	uint64_t now = SystemTimeToMs( GetSystemClockTime() );
	uint64_t delta = now - mLastDeviceChangeCheckTime;
	
	if (delta < 5000)
	{
		return;
	}

	(void)CheckForDeviceChanges();

	mLastDeviceChangeCheckTime = now;
}
