/********************************************************************************************
* Twitch Broadcasting SDK
*
* This software is supplied under the terms of a license agreement with Twitch Interactive, Inc. and
* may not be copied or used except in accordance with the terms of that agreement
* Copyright (c) 2012-2014 Twitch Interactive, Inc.
*********************************************************************************************/

#include "twitchcore/internal/pch.h"
#include "twitchcore/wiresharkreplaysocket.h"
#include "twitchcore/sysclock.h"
#include "twitchcore/thread.h"

#define PACKET_VARIABLE(peer, index) "peer" #peer "_" #index

namespace
{
	#include "c:/drew/junk/replay.inc"
}

//-----------------------------------------------------------------------------
ttv::WiresharkReplaySocket::WiresharkReplaySocket()
:	mTotalSent(0)
,	mTotalRecieved(0)
,	mNextInputPacket(0)
,	mNextOutputPacket(0)
,	mNextInputOffset(0)
,	mConnectTime(0)
,	mBlocking(true)
,	mConnected(false)
{
}

//-----------------------------------------------------------------------------
ttv::WiresharkReplaySocket::~WiresharkReplaySocket()
{
	(void)Disconnect();
}

//-----------------------------------------------------------------------------
TTV_ErrorCode ttv::WiresharkReplaySocket::FlushCache()
{
	// We don't buffer file I/O
	return TTV_EC_SUCCESS;
}


//-----------------------------------------------------------------------------
TTV_ErrorCode ttv::WiresharkReplaySocket::ConnectTo(const std::string& /*host*/, const std::string& /*port*/)
{
	assert(!mConnected);
	if (mConnected)
	{
		return TTV_EC_SOCKET_EALREADY;
	}

	mTotalSent = 0;
	mTotalRecieved = 0;
	mNextInputPacket = 0;
	mNextOutputPacket = 0;
	mNextInputOffset = 0;
	mConnectTime = GetCurrentTimeMilliseconds();
	mConnected = true;

	return TTV_EC_SUCCESS;
}

//-----------------------------------------------------------------------------
TTV_ErrorCode ttv::WiresharkReplaySocket::Disconnect()
{
	// Close files
	mConnected = false;

	return TTV_EC_SUCCESS;
}

//-----------------------------------------------------------------------------
TTV_ErrorCode ttv::WiresharkReplaySocket::TCPListen(const std::string& /*host*/, const std::string& /*port*/)
{
	return TTV_EC_UNIMPLEMENTED;
}

//-----------------------------------------------------------------------------
TTV_ErrorCode ttv::WiresharkReplaySocket::AcceptConnection(ISocket** newSocket)
{
	*newSocket = nullptr;

	return TTV_EC_UNIMPLEMENTED;
}

//-----------------------------------------------------------------------------
TTV_ErrorCode ttv::WiresharkReplaySocket::Send(const uint8_t* buffer, size_t length, bool /*cache*/)
{
	assert (buffer);
	assert (length > 0);
	
	if (!Connected())
	{
		return TTV_EC_SOCKET_ENOTCONN;
	}

	// TODO: Maybe we want to assert that the sent data is the same as captured

	return TTV_EC_SUCCESS;
}

//-----------------------------------------------------------------------------
TTV_ErrorCode ttv::WiresharkReplaySocket::Recv(uint8_t* buffer, size_t length, size_t& received, bool /*fillBuffer*/)
{
	assert (buffer);

	TTV_ErrorCode ret = TTV_EC_SUCCESS;
	received = 0;

	if (Connected())
	{
		// Fill the buffer with receieved packets
		while (received < length)
		{
			PacketData* packet = GetInputPacket(mNextInputPacket);
			if (packet != nullptr)
			{
				// Simulate timing
				uint64_t delta = GetCurrentTimeMilliseconds() - mConnectTime;
				if (delta >= packet->time)
				{
					size_t packetRemaining = packet->length - mNextInputOffset;
					size_t receiveRemaining = length - received;
					size_t copySize = std::min(packetRemaining, receiveRemaining);

					memcpy(&buffer[received], &packet->data[mNextInputOffset], copySize);

					received += copySize;
					mNextInputOffset += copySize;

					// Advance to the next packet
					if (mNextInputOffset == packet->length)
					{
						mNextInputOffset = 0;
						mNextInputPacket++;
					}
				}
				else
				{
					// Data hasn't arrived yet
					if (mBlocking)
					{
						uint wait = static_cast<uint>(packet->time - delta);
						Thread::Sleep(wait);
					}
					else
					{
						break;
					}
				}
			}
			else
			{
				// No more data to receive
				break;
			}
		}

		if (received < length)
		{
			if (mBlocking)
			{
				ret = TTV_EC_SOCKET_ECONNABORTED;
				(void)Disconnect();
			}
		}
	}
	else
	{
		ret = TTV_EC_SOCKET_ENOTCONN;
	}

	return TTV_EC_SUCCESS;
}

//-----------------------------------------------------------------------------
TTV_ErrorCode ttv::WiresharkReplaySocket::SetBlockingMode(bool blockingMode)
{
	mBlocking = blockingMode;

	return TTV_EC_SUCCESS;
}

//-----------------------------------------------------------------------------
uint64_t ttv::WiresharkReplaySocket::TotalSent() const 
{ 
	return mTotalSent;
}

//-----------------------------------------------------------------------------
uint64_t ttv::WiresharkReplaySocket::TotalReceived() const
{ 
	return mTotalRecieved;
}

//-----------------------------------------------------------------------------
bool ttv::WiresharkReplaySocket::Connected() 
{ 
	return mConnected;
}
