package main

import (
	"fmt"
	"sort"
	"testing"

	"github.com/graph-gophers/graphql-go"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestItems(t *testing.T) {
	gqlPointer := func(in string) *graphql.ID {
		gin := graphql.ID(in)
		return &gin
	}

	t.Run("sort.Sort", func(t *testing.T) {
		basicSortedCase := items{
			&item{id: "0"},
			&item{id: "1"},
			&item{id: "2"},
		}

		tcs := []struct {
			Items  items
			Sorted items
		}{
			{
				Items:  items{},
				Sorted: items{},
			},
			{
				Items:  items{&item{id: "0"}},
				Sorted: items{&item{id: "0"}},
			},
			{
				Items:  basicSortedCase,
				Sorted: basicSortedCase,
			},
			{
				Items:  items{&item{id: "2"}, &item{id: "1"}, &item{id: "0"}},
				Sorted: basicSortedCase,
			},
			{
				Items:  items{&item{id: "2"}, &item{id: "0"}, &item{id: "1"}},
				Sorted: basicSortedCase,
			},
		}

		for nTc, tc := range tcs {
			t.Run(fmt.Sprintf("%d", nTc), func(t *testing.T) {
				sort.Sort(tc.Items)
				assert.Equal(t, tc.Sorted, tc.Items)
			})
		}
	})

	t.Run("After", func(t *testing.T) {
		i := items{
			&item{id: "0"},
			&item{id: "1"},
			&item{id: "2"},
		}

		assert.Equal(t, i, i.After(nil))
		assert.Equal(t, i[1:], i.After(gqlPointer("0")))
		assert.Equal(t, i[2:], i.After(gqlPointer("1")))
		assert.Equal(t, items{}, i.After(gqlPointer("2")))
	})

	t.Run("First", func(t *testing.T) {
		i := items{
			&item{id: "0"},
			&item{id: "1"},
			&item{id: "2"},
		}

		assert.Equal(t, items{}, i.First(0))
		assert.Equal(t, i[:1], i.First(1))
		assert.Equal(t, i[:2], i.First(2))
		assert.Equal(t, i[:3], i.First(3))
		assert.Equal(t, i[:3], i.First(4))
	})

	t.Run("Get", func(t *testing.T) {
		i := items{&item{id: "0"}}

		t.Run("found", func(t *testing.T) {
			res, ok := i.Get(graphql.ID("0"))
			require.True(t, ok)
			assert.Equal(t, i[0], res)
		})

		t.Run("not found", func(t *testing.T) {
			_, ok := i.Get(graphql.ID("1"))
			assert.False(t, ok)
		})
	})
}
