import type * as cdk from 'aws-cdk';
import * as aws from 'aws-sdk';
import { exec as callbackExec } from 'child_process';
import * as util from 'util';
import { authenticateWithMidway } from './authenticate-with-midway';
import { requestIsengardCredentials } from './request-isengard-credentials';
import { assertValidAccountID, assertValidRoleName } from './validation';
const homedir = require('os').homedir();
const exec = util.promisify(callbackExec);

class IsengardCredentialProviderSource implements cdk.CredentialProviderSource {
  public name = 'cdk-plugin-isengard';

  public async isAvailable(): Promise<boolean> {
    return true;
  }

  public async canProvideCredentials(_accountId: string): Promise<boolean> {
    return true;
  }

  public async getProvider(accountID: string, _mode: cdk.Mode): Promise<aws.Credentials> {
    assertValidAccountID(accountID);

    const roleName = process.env.ISENGARD_ROLE || 'Admin';
    assertValidRoleName(roleName);

    return requestIsengardCredentials({
      accountID,
      roleName,
      requestCreds: curlIsengardService,
      authenticate: authenticateWithMidway,
    });
  }
}

// Using cURL here since it's a required dependency of mwinit and it reads and
// writes the cookie file in a specific way that other libraries often can't
// replicate.
async function curlIsengardService(accountID: string, roleName: string) {
  const roleData = JSON.stringify({
    AWSAccountID: accountID,
    IAMRoleName: roleName,
  });

  const postBody = process.platform === 'win32'
  ? `"${roleData.replace(/"/g, '\\"')}"`
  : `'${roleData}'`;

  const midwayCookie = `${homedir}/.midway/cookie`;

  const command = [
    'curl',
    `-b ${midwayCookie}`,
    `-c ${midwayCookie}`,
    '-L',
    '-X POST',
    '--header "X-Amz-Target: IsengardService.GetAssumeRoleCredentials"',
    '--header "Accept: application/json"',
    '--header "Content-Encoding: amz-1.0"',
    '--header "Content-Type: application/json; charset=UTF-8"',
    `-d ${postBody}`,
    'https://isengard-service.amazon.com',
  ].join(' ');

  const { stdout } = await exec(command);

  return stdout;
}

export = {
  version: '1',
  init(host: cdk.PluginHost): void {
    host.registerCredentialProviderSource(new IsengardCredentialProviderSource());
  },
};
