import { HeadObjectCommand, S3Client } from "@aws-sdk/client-s3";
import { Construct, RemovalPolicy, Stack } from "monocdk";
import * as iam from "monocdk/aws-iam";
import { CfnEnvironment as MWAAEnvironment } from "monocdk/aws-mwaa";
import * as s3 from "monocdk/aws-s3";

export async function getLatestPluginVersion(
  bucket: string,
  key: string
): Promise<string> {
  const client = new S3Client({ region: "us-west-2" });
  const command = new HeadObjectCommand({
    Bucket: bucket,
    Key: key,
  });

  const results = await client.send(command);
  return results.VersionId!;
}

export interface MwaaAuxiliaryResourcesProps {
  readonly prefix: string;
}

export class MwaaAuxiliaryResources extends Construct {
  readonly prefix: string;
  readonly role: iam.Role;
  readonly bucket: s3.Bucket;
  readonly bucketName: string;

  constructor(
    scope: Construct,
    id: string,
    props: MwaaAuxiliaryResourcesProps
  ) {
    super(scope, id);
    this.prefix = props.prefix;
    this.bucketName = `${props.prefix}-mwaa`;
    this.bucket = new s3.Bucket(this, `${props.prefix}-bucket`, {
      bucketName: this.bucketName,
      removalPolicy: RemovalPolicy.DESTROY,
      autoDeleteObjects: true,
      blockPublicAccess: {
        blockPublicAcls: true,
        blockPublicPolicy: true,
        ignorePublicAcls: false,
        restrictPublicBuckets: false,
      },
      versioned: true,
    });

    const region = Stack.of(this).region;
    const account = Stack.of(this).account;

    // Airflow Execution Role
    this.role = new iam.Role(this, `${props.prefix}-execution-role`, {
      assumedBy: new iam.CompositePrincipal(
        new iam.ServicePrincipal("airflow-env.amazonaws.com"),
        new iam.ServicePrincipal("airflow.amazonaws.com")
      ),
      managedPolicies: [
        iam.ManagedPolicy.fromAwsManagedPolicyName("AmazonSageMakerFullAccess"),
      ],
    });
    const policyDocument = new iam.PolicyDocument();
    const airflowMetricsStatement = new iam.PolicyStatement({
      effect: iam.Effect.ALLOW,
    });
    airflowMetricsStatement.addActions("airflow:PublishMetrics");
    airflowMetricsStatement.addResources("*");
    const cloudwatchStatement = new iam.PolicyStatement({
      effect: iam.Effect.ALLOW,
    });
    cloudwatchStatement.addActions(
      "logs:CreateLogStream",
      "logs:CreateLogGroup",
      "logs:PutLogEvents",
      "logs:GetLogEvents",
      "logs:GetLogRecord",
      "logs:GetLogGroupFields",
      "logs:GetQueryResults",
      "logs:DescribeLogGroups",
      "cloudwatch:PutMetricData"
    );
    cloudwatchStatement.addResources("*");
    const s3Statement = new iam.PolicyStatement({ effect: iam.Effect.ALLOW });
    s3Statement.addActions("s3:GetObject*", "s3:GetBucket*", "s3:List*");
    s3Statement.addResources(
      this.bucket.bucketArn,
      this.bucket.arnForObjects("*")
    );
    const secretsManagerStatement = new iam.PolicyStatement({
      effect: iam.Effect.ALLOW,
    });
    secretsManagerStatement.addActions(
      "secretsmanager:GetResourcePolicy",
      "secretsmanager:GetSecretValue",
      "secretsmanager:DescribeSecret",
      "secretsmanager:ListSecretVersionIds"
    );
    secretsManagerStatement.addResources("*");
    const sqsStatement = new iam.PolicyStatement({ effect: iam.Effect.ALLOW });
    sqsStatement.addActions(
      "sqs:ChangeMessageVisibility",
      "sqs:DeleteMessage",
      "sqs:GetQueueAttributes",
      "sqs:GetQueueUrl",
      "sqs:ReceiveMessage",
      "sqs:SendMessage"
    );
    sqsStatement.addResources(`arn:aws:sqs:${region}:*:airflow-celery-*`);
    const kmsStatement = new iam.PolicyStatement({ effect: iam.Effect.ALLOW });
    kmsStatement.addActions(
      "kms:Decrypt",
      "kms:DescribeKey",
      "kms:GenerateDataKey*",
      "kms:Encrypt"
    );
    kmsStatement.addNotResources(`arn:aws:kms:*:${account}:key/*`);
    kmsStatement.addCondition("StringLike", {
      "kms:ViaService": `sqs.${region}.amazonaws.com`,
    });
    policyDocument.addStatements(
      airflowMetricsStatement,
      cloudwatchStatement,
      s3Statement,
      secretsManagerStatement,
      sqsStatement,
      kmsStatement
    );
    new iam.Policy(this, `${props.prefix}-execution-role-policy`, {
      document: policyDocument,
    }).attachToRole(this.role);
  }
}

export interface ConductorMwaaEnvironmentProps {
  readonly auxiliaryResources: MwaaAuxiliaryResources;
  readonly subnetIds: Array<string>;
  readonly securityGroupIds: Array<string>;
  readonly airflowVersion: string;
  readonly requirementsS3ObjectVersion: string;
  readonly pluginsS3ObjectVersion: string;
  readonly webserverAccessMode?: string;
  readonly weeklyMaintenanceWindowStart?: string;
  readonly maxWorkers?: number;
  readonly environmentClass?: string;
}

export class ConductorMwaaClusterEnvironment extends Construct {
  mwaaEnvironment: MWAAEnvironment;

  constructor(
    scope: Construct,
    id: string,
    props: ConductorMwaaEnvironmentProps
  ) {
    super(scope, id);
    this.mwaaEnvironment = new MWAAEnvironment(
      this,
      `${props.auxiliaryResources.prefix}-airflow-cluster`,
      {
        name: props.auxiliaryResources.prefix,
        airflowVersion: props.airflowVersion,
        dagS3Path: "dags",
        executionRoleArn: props.auxiliaryResources.role.roleArn,
        requirementsS3Path: "requirements.txt",
        requirementsS3ObjectVersion: props.requirementsS3ObjectVersion,
        pluginsS3Path: "plugins.zip",
        pluginsS3ObjectVersion: props.pluginsS3ObjectVersion,
        sourceBucketArn: props.auxiliaryResources.bucket.bucketArn,
        maxWorkers: props.maxWorkers,
        environmentClass: props.environmentClass,
        networkConfiguration: {
          subnetIds: props.subnetIds,
          securityGroupIds: props.securityGroupIds,
        },
        webserverAccessMode: props.webserverAccessMode || "PUBLIC_ONLY",
        weeklyMaintenanceWindowStart:
          props.weeklyMaintenanceWindowStart || "WED:16:00", // 4 PM UTC, 8 AM PST, 9 AM PDT
        // Use Secrets Manager for the provider back end.
        airflowConfigurationOptions: {
          "secrets.backend":
            "airflow.providers.amazon.aws.secrets.secrets_manager.SecretsManagerBackend",
          "secrets.backend_kwargs": `{"connections_prefix" : "airflow/connections", "variables_prefix" : "airflow/variables"}`,
        },
        // Default to maximum logging.
        loggingConfiguration: {
          dagProcessingLogs: { enabled: true, logLevel: "INFO" },
          schedulerLogs: { enabled: true, logLevel: "INFO" },
          taskLogs: { enabled: true, logLevel: "INFO" },
          webserverLogs: { enabled: true, logLevel: "INFO" },
          workerLogs: { enabled: true, logLevel: "INFO" },
        },
      }
    );
  }
}
