import autoscaling = require("@aws-cdk/aws-autoscaling");
import cdk = require("@aws-cdk/core");
import iam = require("@aws-cdk/aws-iam");

export interface CfnInitConfig {
  /**
   * onLaunch config is applied when a new machine is launched by the
   * autoscaling group.
   */
  onLaunch?: CfnInitConfigPhase;

  /**
   * onUpdate config is applied whenever the metadata of the launchConfig
   * changes. In other words, whenever config is updated by CloudFormation,
   * this config takes effect.
   */
  onUpdate?: CfnInitConfigPhase;

  /**
   * Run both at launch time and whenever CloudFormation updates config.
   */
  always?: CfnInitConfigPhase;

  /**
   * A callback to allow config to add new access that the EC2 instances may
   * need.
   */
  addAccess?: (role: iam.IRole) => void;
}

export interface CfnInitConfigPhase {
  name: string;
  files?: Array<{
    path: string;
    content: string[] | {} | string;
    mode?: string;
  }>;
  commands?: Array<{
    name: string;
    command: string;
  }>;
}

interface CfnInitConfigPayload {
  [name: string]: {
    files?: {
      [path: string]: {
        content: string;
        mode: string;
        owner: string;
        group: string;
      };
    };
    commands?: {
      [name: string]: {
        command: string;
      };
    };
  };
}

interface CfnInitProps {
  autoScalingGroup: autoscaling.AutoScalingGroup;
  configs?: CfnInitConfig[];
}

/**
 * CfnInit is a helper class for using cfn-init
 * https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cfn-init.html
 */
export class CfnInit {
  private configs: CfnInitConfig[] = [];
  private sg: autoscaling.AutoScalingGroup;
  private launchConfig: autoscaling.CfnLaunchConfiguration;

  constructor(props: CfnInitProps) {
    this.sg = props.autoScalingGroup;
    this.launchConfig = this.sg.node.findChild(
      "LaunchConfig"
    ) as autoscaling.CfnLaunchConfiguration;
    this.bootstrapCfnInit();
    if (props.configs) {
      this.configs.push(...props.configs);
    }

    this.launchConfig.addOverride("Metadata", this.toJSON());
  }

  public toJSON() {
    const configSets = {
      default: [] as string[],
      UpdateEnvironment: [] as string[]
    };

    const configEntries: CfnInitConfigPayload = {};

    // Allow the config to add any needed access to the instance role.
    for (const config of this.configs) {
      config.addAccess?.(this.sg.role);
    }

    // The configSets order commands and configuration so that they run in a
    // deterministic order. ConfigSets also group commands to determine whether
    // they run on boot and/or Cfn update.
    for (const config of this.configs) {
      if (config.onLaunch) {
        configSets.default.push(config.onLaunch.name);
      }
      if (config.always) {
        configSets.default.push(config.always.name);
        configSets.UpdateEnvironment.push(config.always.name);
      }
      if (config.onUpdate) {
        configSets.UpdateEnvironment.push(config.onUpdate.name);
      }
    }

    // Gather all configs from the different lifecycles.
    const allLifecycleConfigs = [];
    for (const config of this.configs) {
      if (config.onLaunch) {
        allLifecycleConfigs.push(config.onLaunch);
      }
      if (config.always) {
        allLifecycleConfigs.push(config.always);
      }
      if (config.onUpdate) {
        allLifecycleConfigs.push(config.onUpdate);
      }
    }

    for (const config of allLifecycleConfigs) {
      configEntries[config.name] = {};
      if (config.files) {
        const files: {
          [path: string]: {
            content: string;
            mode: string;
            owner: string;
            group: string;
          };
        } = {};

        for (const file of config.files) {
          let contentString: string;
          if (typeof file.content === 'string') {
            contentString = file.content;
          } else if (Array.isArray(file.content)) {
            contentString = file.content.join("\n");
          } else {
            contentString = JSON.stringify(file.content, null, 2);
          }

          files[file.path] = {
            content: cdk.Fn.sub(contentString),
            group: "root",
            owner: "root",
            mode: file.mode ?? "000400"
          };
        }

        configEntries[config.name].files = files;
      }

      if (config.commands) {
        const commands: {
          [name: string]: {
            command: string;
          };
        } = {};

        for (const [i, command] of config.commands.entries()) {
          const name = `${(i + 1).toString().padStart(2, "0")}_${command.name}`;
          commands[name] = {
            command: command.command
          };
        }

        configEntries[config.name].commands = commands;
      }
    }

    const payload = {
      "AWS::CloudFormation::Init": {
        configSets,
        ...configEntries
      }
    };

    return payload;
  }

  /**
   * Add configuration for cfn-init so that it invokes config commands.
   *
   * This configuration is adapted from:
   * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/Install-CloudWatch-Agent-New-Instances-CloudFormation.html
   */
  private bootstrapCfnInit() {
    this.sg.role.addManagedPolicy(
      iam.ManagedPolicy.fromAwsManagedPolicyName("AmazonSSMManagedInstanceCore")
    );
    const resource = this.launchConfig.logicalId;

    this.sg.addUserData(
      cdk.Fn.sub(
        [
          // Setup the SSM Agent
          "yum install -y https://s3.amazonaws.com/ec2-downloads-windows/SSMAgent/latest/linux_amd64/amazon-ssm-agent.rpm",
          "sudo systemctl enable amazon-ssm-agent",

          // Install CfnInit because it's not included in the ECS optimized Linux distro
          "yum install -y aws-cfn-bootstrap",

          // Initialize CfnInit
          `/opt/aws/bin/cfn-init -v --stack \${AWS::StackId} --resource ${resource} --region \${AWS::Region} --configsets default`,
          `/opt/aws/bin/cfn-signal -e $? --stack \${AWS::StackId} --resource ${resource} --region \${AWS::Region}`
        ].join("\n")
      )
    );

    this.configs.push({
      onLaunch: {
        name: "setup-cfn-hup",
        files: [
          {
            path: "/etc/cfn/cfn-hup.conf",
            content: [
              "[main]",
              "stack=${AWS::StackId}",
              "region=${AWS::Region}",
              "interval=1"
            ]
          },
          {
            path: "/etc/cfn/hooks.d/cfn-init-auto-reloader.conf",
            content: [
              "[cfn-auto-reloader-hook]",
              "triggers=post.update",
              `path=Resources.${resource}.Metadata.AWS::CloudFormation::Init`,
              `action=/opt/aws/bin/cfn-init -v --stack \${AWS::StackId} --resource ${resource} --region \${AWS::Region} --configsets UpdateEnvironment`,
              "runas=root"
            ]
          }
        ],
        commands: [
          {
            name: "enable-cfn-hup",
            command: "systemctl enable cfn-hup.service"
          },
          {
            name: "start-cfn-hup",
            command: "systemctl start cfn-hup.service"
          }
        ]
      }
    });
  }
}

export * from './cloudwatch-agent';
export * from './docker-disk-metrics';
