import path = require('path');
import cdk = require('@aws-cdk/core');
import ec2 = require('@aws-cdk/aws-ec2');
import autoscaling = require('@aws-cdk/aws-autoscaling');
import ecs = require('@aws-cdk/aws-ecs');
import elbv2 = require('@aws-cdk/aws-elasticloadbalancingv2');
import r53 = require('@aws-cdk/aws-route53');
import r53_targets = require('@aws-cdk/aws-route53-targets');
import iam = require('@aws-cdk/aws-iam');
import { Duration } from "@aws-cdk/core";
import { CfnInit, cloudWatchAgentConfig, dockerDiskMetrics } from "./cfn-init";

const JOB_MANAGER_INSTANCE_TYPE = 'm4.large';

export enum FlinkVersion {
  V1_10 = "1.10.1",
  V1_11 = "1.11.2",
}

interface ClusterProps {
  vpc: ec2.IVpc;
  dnsNamespace?: string;
  environment?: { [key: string]: string };
  flinkConf: { [key: string]: string };
  taskManagerInstances: number;
  taskManagerInstanceType: string;
  taskManagerSlots: number;
  jobManagerInstances: number;
  loadBalancerName?: string;
  flinkVersion?: FlinkVersion;
  customContainerImage?: ecs.ContainerImage;
}

export class Cluster extends cdk.Construct {
  public readonly taskManagerDefinition: ecs.Ec2TaskDefinition;
  public readonly jobManagerDefinition: ecs.Ec2TaskDefinition;
  public readonly alb: elbv2.ApplicationLoadBalancer;
  public readonly taskManagerASG: autoscaling.AutoScalingGroup;
  public readonly jobManagerASG: autoscaling.AutoScalingGroup;

  constructor(scope: cdk.Construct, id: string, props: ClusterProps) {
    super(scope, id);

    const amazonS3FullAccess = iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonS3FullAccess');

    const cluster = new ecs.Cluster(this, 'FlinkCluster', { vpc: props.vpc });
    this.jobManagerASG = cluster.addCapacity('FlinkJobManager', {
      instanceType: new ec2.InstanceType(JOB_MANAGER_INSTANCE_TYPE),
      minCapacity: props.jobManagerInstances,
    });
    this.taskManagerASG = cluster.addCapacity('FlinkTaskManagers', {
      instanceType: new ec2.InstanceType(props.taskManagerInstanceType),
      minCapacity: props.taskManagerInstances,
    });


    new CfnInit({
      autoScalingGroup: this.jobManagerASG,
      configs: [cloudWatchAgentConfig(), dockerDiskMetrics()],
    });

    new CfnInit({
      autoScalingGroup: this.taskManagerASG,
      configs: [cloudWatchAgentConfig(), dockerDiskMetrics()],
    });

    let flinkProperties = "";
    Object.entries(props.flinkConf).forEach(([key, value]) => {
      flinkProperties = flinkProperties.concat(`${key}: ${value}\n`);
    });

    // Create job managers
    const masterTaskDefinition = new ecs.Ec2TaskDefinition(this, 'FlinkMaster', {
      networkMode: ecs.NetworkMode.AWS_VPC,
    });
    this.jobManagerDefinition = masterTaskDefinition;
    masterTaskDefinition.taskRole.addManagedPolicy(amazonS3FullAccess);

    const buildArgs: Record<string, string> = {};
    if (props.flinkVersion) {
      buildArgs["FLINK_VERSION"] = props.flinkVersion;
    }

    const jobManagerContainer = masterTaskDefinition.addContainer("JobManagerContainer", {
      image: props.customContainerImage || ecs.ContainerImage.fromAsset(path.join(__dirname, 'image'), {
        buildArgs
      }),
      command: ['jobmanager'],
      memoryReservationMiB: 512,
      logging: new ecs.AwsLogDriver({
        logRetention: 30,
        streamPrefix: 'flink-job-manager',
      }),
      environment: {
        'FLINK_PROPERTIES': flinkProperties,
        ...(props.environment || {}),
      },
    });

    const jobManagerService = new ecs.Ec2Service(this, 'JobManagerService', {
      cluster: cluster,
      taskDefinition: masterTaskDefinition,
      desiredCount: props.jobManagerInstances,
      placementConstraints: [
        ecs.PlacementConstraint.memberOf(`attribute:ecs.instance-type == ${JOB_MANAGER_INSTANCE_TYPE}`),
        ecs.PlacementConstraint.distinctInstances(),
      ],
    });

    const jobManagerExposedPorts = [8081, 6123, 6124, 6125, 50102];
    jobManagerExposedPorts.forEach(port => {
      jobManagerContainer.addPortMappings({ containerPort: port, hostPort: port });
      jobManagerService.connections.allowFromAnyIpv4(ec2.Port.tcp(port));
    });

    // Create task managers
    const taskManagerDefinition = new ecs.Ec2TaskDefinition(this, 'FlinkTaskManager', {
      networkMode: ecs.NetworkMode.AWS_VPC,
    });
    this.taskManagerDefinition = taskManagerDefinition;
    taskManagerDefinition.taskRole.addManagedPolicy(amazonS3FullAccess);
    taskManagerDefinition.taskRole.addToPolicy(
      new iam.PolicyStatement({
        actions: [
          'cloudwatch:PutMetricData',
        ],
        resources: ['*'],
      }),
    );

    const taskManagerContainer = taskManagerDefinition.addContainer("TaskManagerContainer", {
      image: props.customContainerImage || ecs.ContainerImage.fromAsset(path.join(__dirname, 'image'), {
        buildArgs
      }),
      command: ['taskmanager'],
      // memoryReservationMiB, subtracted from the total instance memory, is used to estimate taskmanager.memory.process.size in flink-conf.yaml
      memoryReservationMiB: 512,
      logging: new ecs.AwsLogDriver({
        logRetention: 30,
        streamPrefix: 'flink-task-manager',
      }),
      environment: {
        FLINK_PROPERTIES: flinkProperties,
        // Reserve one CPU for non-task work like checkpointing
        TASK_MANAGER_NUMBER_OF_TASK_SLOTS: `${props.taskManagerSlots}`,
        ...(props.environment || {}),
      },
    });
    const taskManagerService = new ecs.Ec2Service(this, 'TaskManagerService', {
      cluster: cluster,
      taskDefinition: taskManagerDefinition,
      desiredCount: props.taskManagerInstances,
      placementConstraints: [
        ecs.PlacementConstraint.memberOf(`attribute:ecs.instance-type == ${props.taskManagerInstanceType}`),
        ecs.PlacementConstraint.distinctInstances(),
      ],
    });

    const taskManagerExposedPorts = [6121, 6122, 50100, 50101];
    taskManagerExposedPorts.forEach(port => {
      taskManagerContainer.addPortMappings({ containerPort: port, hostPort: port });
      taskManagerService.connections.allowFromAnyIpv4(ec2.Port.tcp(port));
    });

    const lb = new elbv2.ApplicationLoadBalancer(this, 'FlinkClusterLB', { vpc: props.vpc, internetFacing: false, loadBalancerName: props.loadBalancerName });
    this.alb = lb;
    const webListener = lb.addListener('WebListener', {
      port: 8081,
      protocol: elbv2.ApplicationProtocol.HTTP,
    });
    webListener.addTargets('FlinkWebRouter', {
      protocol: elbv2.ApplicationProtocol.HTTP,
      targets: [jobManagerService],
      healthCheck: { path: '/' },
      stickinessCookieDuration: Duration.days(1),
    });

    const privateZone = new r53.PrivateHostedZone(this, "HostedZone", {
      zoneName: !props.dnsNamespace ? 'job-manager.flink' : `job-manager.flink.${props.dnsNamespace}`,
      vpc: props.vpc,
    });

    const target = new r53_targets.LoadBalancerTarget(lb);
    new r53.ARecord(this, 'FlinkRecord', {
      zone: privateZone,
      recordName: !props.dnsNamespace ? 'job-manager.flink' : `job-manager.flink.${props.dnsNamespace}`,
      target: r53.AddressRecordTarget.fromAlias(target),
    });
  }
}
