import cdk = require('@aws-cdk/core');
import lambda = require('@aws-cdk/aws-lambda');
import iam = require('@aws-cdk/aws-iam');
import ec2 = require('@aws-cdk/aws-ec2');
import events = require("@aws-cdk/aws-events");
import eventsTargets = require("@aws-cdk/aws-events-targets");
import path = require('path');
import { Repo } from './repo';

interface CleanerProps {
  /** 
   * A Github repo that the cleaner should use to find dependent CloudFormation
   * stacks to destroy.
   */
  repo: Repo;

  /**
   * The most common usage of CloudFormation stacks requires access to all the
   * underlying resources managed by CloudFormation. Because of this,
   * CloudFormation users typically have the Admin role. However, there are
   * more granular ways to handle access. CloudFormation stacks can be given
   * service roles or you can give the Lambda just the access needed with the
   * `addToRolePolicy` method.
   * 
   * The value defaults to `false` to make sure users are aware of the access
   * this construct has.
   * 
   * @default false
   */
  adminPermissions?: boolean,

  /**
   * A schedule to determine how often to run the cleanup process.
   * 
   * @default Every hour on the hour: 0 * * * *
   */
  schedule?: events.Schedule,

  /**
   * If your Github endpoint can only be reached from inside your VPC,
   * then you'll need to provide a VPC to attach the lambda to.
   */
  vpc?: ec2.IVpc;
}

export class Cleaner extends cdk.Construct {
  private fn: lambda.Function;

  constructor(scope: cdk.Construct, name: string, props: CleanerProps) {
    super(scope, name);

    const codePath = path.join(__dirname, 'handler', 'bundle.zip');

    this.fn = new lambda.Function(this, 'Resource', {
      vpc: props.vpc,
      runtime: lambda.Runtime.NODEJS_14_X,
      code: lambda.Code.fromAsset(codePath),
      handler: 'dist/index.handler',
      timeout: cdk.Duration.minutes(15),
      environment: {
        GITHUB_URL: props.repo.url,
        GITHUB_AUTH: props.repo.auth.toString(),
        GITHUB_OWNER: props.repo.owner,
        GITHUB_REPO: props.repo.name,
      }
    });

    if (props.adminPermissions) {
      this.fn.addToRolePolicy(new iam.PolicyStatement({
        actions: ['*'],
        resources: ['*'],
      }));
    }

    // By default run the Lambda on the hour
    const schedule = props.schedule ?? events.Schedule.cron({ minute: '0' });

    const cronInvocation = new events.Rule(this, "CronInvocation", {
      schedule,
    });
    cronInvocation.addTarget(new eventsTargets.LambdaFunction(this.fn));
  }

  /**
   * Add your own policy statement. This is a proxy to the underlying Lambda's
   * `addToRolePolicy` method.
   */
  public addToRolePolicy(statement: iam.PolicyStatement) {
    this.fn.addToRolePolicy(statement);
  }
}
