import { Octokit } from '@octokit/rest';
import * as aws from 'aws-sdk';

const env = {
  githubAuth: mustGetENV('GITHUB_AUTH'),
  githubOwner: mustGetENV('GITHUB_OWNER'),
  githubRepo: mustGetENV('GITHUB_REPO'),
  githubURL: mustGetENV('GITHUB_URL'),
}

/**
 * Manual testing
 * 1. Copy paste Isengard credentials from test account.
 * 2. Run handler `AWS_REGION=us-west-2 npx ts-node lib/temp-stack-cleaner/handler.ts`
 */
async function handler(): Promise<void> {
  const [tempStacks, branchNames] = await Promise.all([
    getTempStacks(),
    getBranchNames(),
  ]);
  const doomedStacks = tempStacks.filter(stack => !branchNames.has(stack.branch));

  const cfn = new aws.CloudFormation();
  for (const stack of doomedStacks) {
    log.info({ message: 'Deleting stack', stackName: stack.name });

    try {
      await cfn.deleteStack({ StackName: stack.name }).promise();
    } catch(e) {
      log.error({
        message: 'Failed to request stack deletion',
        stackName: stack.name,
        error: e.message,
      });
    }
  }
}

/**
 * Find all the CloudFormation stacks with the required github tags.
 */
async function getTempStacks() {
  const tagAPI = new aws.ResourceGroupsTaggingAPI();
  const response = await tagAPI.getResources({
    ResourceTypeFilters: ['cloudformation:stack'],
    TagFilters: [{
      Key: 'github-owner',
      Values: [env.githubOwner],
    }, {
      Key: 'github-repo',
      Values: [env.githubRepo],
    }, {
      Key: 'github-branch',
    }],
  }).promise();

  const stacks = response.ResourceTagMappingList;
  if (!stacks || stacks.length === 0) {
    log.info({ message: 'No temporary stacks found.' });
    return [];
  }

  const tempStacks = [];
  for (const stack of stacks) {
    if (!stack.ResourceARN) {
      log.warn({
        message: "Missing stack property. Skipping.",
        property: "ResourceARN"
      });
      continue;
    }

    if (!stack.Tags) {
      log.warn({
        message: "Missing stack property. Skipping.",
        property: "Tags"
      });
      continue;
    }

    const tagMap = new Map<string, string>();
    stack.Tags.forEach(t => tagMap.set(t.Key, t.Value));

    const branch = tagMap.get('github-branch');
    if (!branch) {
      log.error({
        message: "Failed to get the github-branch value from stack",
        stack: stack.ResourceARN
      });
      continue;
    }

    const name = stack.ResourceARN.split('/')[1];
    tempStacks.push({ name, branch });
  }

  return tempStacks;
}

/**
 * Return a set of all the branch names in the GitHub repo.
 */
async function getBranchNames() {
  const octokit = new Octokit({
    auth: env.githubAuth,
    baseUrl: env.githubURL,
  });

  let prs;
  try {
    prs = await octokit.repos.listBranches({
      repo: env.githubRepo,
      owner: env.githubOwner,
    });
  } catch(e) {
    return log.fatal({
      message: 'Failed fetching the list of Github branches',
      error: e.message,
    });
  }

  const branchNames = new Set(prs.data.map(pr => pr.name));

  return branchNames;
}

function mustGetENV(key: string): string {
  const value = process.env[key];
  if (!value) {
    return log.fatal({
      message: `Missing environment variable.`,
      key,
    });
  }

  return value;
}

interface LogParams {
  [key: string]: any;
  message: string;
}

const log = {
  info(params: LogParams) {
    console.info(params)
  },
  warn(params: LogParams) {
    console.warn(params)
  },
  error(params: LogParams) {
    console.error(params)
  },
  fatal(params: LogParams) {
    console.error(params)
    throw new Error(params.message);
  }
}

exports.handler = handler;
