# Spade Consumer

CDK constructs for consuming [Spade](https://wiki.twitch.com/display/DI/Spade) data. 

## Installation

Install with:

```
npm i @twitch-cdk/spade-consumer
```

CDK was originally designed as a series of small packages, but behaves as a
single library with regard to semantic versioning. Because of this design,
consumers of CDK libraries need to install all third-party library CDK
dependencies in the host. This is an npm concept called "peer dependencies".
If you run into versioning issues, install this library's CDK peer
dependencies into your CDK application.

You can do this in your CDK app with this line (replace <your-cdk-version>
with your pinned version of CDK):

```
V=<your-cdk-version>; npm i --save-exact @aws-cdk/core@$V @aws-cdk/aws-cloudwatch-actions@$V @aws-cdk/aws-kinesis@$V @aws-cdk/aws-sns@$V @aws-cdk/aws-sns-subscriptions@$V
```

Afterward you may need to run this command to get npm to recalculate your
dependencies:

```
rm -rf node_modules && rm package-lock.json && npm i
```

## KinesisStream

This construct creates a Kinesis stream for incoming Spade data. It's based
on a [Terraform
template](https://git-aws.internal.justin.tv/twitch/docs/blob/master/datainfra/kinesis/no_kinsumer.tf).

Here's an example stack that uses the `KinesisStream` construct:

```ts
import * as spade from '@twitch-cdk/spade-consumer';

interface MyStackProps extends cdk.StackProps {
  /**
   * You'll probably want to pass in your account name to build the DI-preferred
   * stream name.
   */
  accountAlias: string;

  /**
   * If you want to know about default alarms, add a topic for your team.
   */
  pagerDutyTopic: sns.ITopic;
}

export class MyStack extends cdk.Stack {
  /**
   * You may wish to expose the stream to allow other constructs to call
   * streamName and grantRead.
   */
  public myStream: spade.IKinesisStream;

  constructor(scope: cdk.Construct, id: string, props: MyStackProps) {
    this.myStream = new spade.KinesisStream(this, "MyStream", {
      streamName: `spade-downstream-prod-${props.accountAlias}-${this.region}-timeseries-rollup`,
      alarmsEnabled: true,
      shardCount: 32,
      additionalAlarmTopics: [pagerDutyTopic],
    });
  }
}
```

Once your stream is deployed, visit the [Data Infra Service
Desk](https://jira.twitch.com/servicedesk/customer/portal/8) to request Spade
data for your new Kinesis stream.
