import * as ec2 from '@aws-cdk/aws-ec2';
import * as elbv2 from '@aws-cdk/aws-elasticloadbalancingv2';
import * as elbv2_targets from '@aws-cdk/aws-elasticloadbalancingv2-targets';
import * as iam from '@aws-cdk/aws-iam';
import * as lambda from '@aws-cdk/aws-lambda';
import * as logs from '@aws-cdk/aws-logs';
import * as redshift from '@aws-cdk/aws-redshift';
import * as cdk from '@aws-cdk/core';
import * as cr from '@aws-cdk/custom-resources';
import * as path from 'path';
import { TahoeProd, TahoeStage } from '.';

const sciEngAccount = '673385534282';

export interface EndpointServiceProps {
  cluster: redshift.Cluster,
  vpc: ec2.IVpc,
  principals: Array<string>;
  tahoeStage?: TahoeStage;
  subnets?: ec2.SubnetSelection;
}

export class EndpointService extends cdk.Construct {
  public readonly serviceName: string;

  constructor(scope: cdk.Construct, id: string, props: EndpointServiceProps) {
    super(scope, id);

    const onEventHandler = new lambda.SingletonFunction(this, 'CustomResourceHandler', {
      uuid: '6233546D-A37B-435B-8C87-33DE7D2E8913',
      code: lambda.Code.fromAsset(path.join(__dirname, 'handler')),
      handler: 'get_leader.handler',
      timeout: cdk.Duration.seconds(10),
      runtime: lambda.Runtime.PYTHON_3_8
    });

    onEventHandler.addToRolePolicy(new iam.PolicyStatement({
      actions: ['redshift:DescribeClusters'],
      resources: ['*'],
    }));

    const { serviceToken } = new cr.Provider(this, 'Provider', {
      onEventHandler,
      logRetention: logs.RetentionDays.ONE_MONTH,
    });

    const leaderResource = new cdk.CustomResource(this, 'Resource', {
      serviceToken,
      properties: {
        RedshiftClusterId: props.cluster.clusterName,
      },
    });

    const nlb = new elbv2.NetworkLoadBalancer(this, 'NLB', {
      vpc: props.vpc,
      vpcSubnets: props.subnets,
    });
    const nlbListener = nlb.addListener("NLBListener", { port: 5439 });
    const nlbTargetGroup = new elbv2.NetworkTargetGroup(this, 'NLBTargetGroup', {
      vpc: props.vpc,
      port: 5439,
      targetType: elbv2.TargetType.IP,
    });
    nlbListener.addTargetGroups('TapTargetGroup', nlbTargetGroup)

    const ipTarget = new elbv2_targets.IpTarget(leaderResource.getAttString('leaderIP'), 5439)
    nlbTargetGroup.addTarget(ipTarget)

    const tahoeAccount = (props.tahoeStage ?? TahoeProd).account; 
    const defaultPrincipals = [
      new iam.AccountPrincipal(tahoeAccount),
      new iam.AccountPrincipal(sciEngAccount),
    ]
    const principals = props.principals.map((p) => new iam.AccountPrincipal(p))
    const endpointService = new ec2.VpcEndpointService(this, 'EndpointService', {
      vpcEndpointServiceLoadBalancers: [nlb],
      acceptanceRequired: false,
      whitelistedPrincipals: defaultPrincipals.concat(principals),
    });

    this.serviceName = endpointService.vpcEndpointServiceName;
  }
}
