import * as iam from '@aws-cdk/aws-iam';
import * as events from '@aws-cdk/aws-events';
import * as targets from '@aws-cdk/aws-events-targets';
import * as cdk from '@aws-cdk/core';
import { TahoeProd, TahoeStage } from '.';

export interface EventRulesProps {
  tahoeStage?: TahoeStage;
  tapId: string;
  adminRoleArn: string
}

export class EventRules extends cdk.Construct {
  constructor(scope: cdk.Construct, id: string, props: EventRulesProps) {
    super(scope, id);
    const account = cdk.Stack.of(this).account;
    const region = cdk.Stack.of(this).region;
    const tahoeAccount = (props.tahoeStage ?? TahoeProd).account;
    // Create an event rule to filter Redshift local-views related events
    const rule = new events.Rule(this, 'TahoeLocalViewsRule', {
      ruleName: `local-views-${props.tapId}-exec-dtls`,
      description:
          `Catch tahoe local view execution details for tap: ${props.tapId} received from redshift API and forward it to Tahoe API event bus`,
      enabled: true,
      eventBus: events.EventBus.fromEventBusArn(this,
          'default', `arn:aws:events:${region}:${account}:event-bus/default`),
      eventPattern: {
        source: ['aws.redshift-data'],
        detail: {
          'statementName' : [{
            prefix: `local-views-${props.tapId}`
          }]
        }
      },
      targets: [new targets.EventBus(events.EventBus.fromEventBusArn(this, 'tahoe-local-views',
        `arn:aws:events:us-west-2:${tahoeAccount}:event-bus/tahoe-local-views`), {
        role: iam.Role.fromRoleArn(this, 'TapAdminRole', props.adminRoleArn)
      })]
    });
  }
}
