import * as iam from '@aws-cdk/aws-iam';
import * as lambda from '@aws-cdk/aws-lambda';
import * as logs from '@aws-cdk/aws-logs';
import * as cdk from '@aws-cdk/core';
import * as cr from '@aws-cdk/custom-resources';
import * as path from 'path';
import { TahoeProd, TahoeStage } from '.';

interface ResourceHandlerProps {
  tahoeStage?: TahoeStage;
}

export class ResourceHandler extends cdk.Construct {
  public readonly serviceToken: string
  constructor(scope: cdk.Construct, name: string, props?: ResourceHandlerProps) {
    super(scope, name)
    const account = cdk.Stack.of(this).account;

    const tahoeStage = props?.tahoeStage ?? TahoeProd;
    const onEventHandler = new lambda.SingletonFunction(this, 'Resource', {
      uuid: 'a3e23c59-6c5a-47d7-a400-9b51fe182610',
      code: lambda.Code.fromAsset(path.join(__dirname, 'handler')),
      handler: 'tahoe.handler',
      timeout: cdk.Duration.minutes(6),
      runtime: lambda.Runtime.PYTHON_3_8,
      lambdaPurpose: "Tahoe Custom Resource",
      environment: {
        REGISTRATION_ROLE: tahoeStage.apiInvocationRoleArn,
        LOGGING_BUCKET: tahoeStage.loggingBucketName,
      }
    });

    onEventHandler.addToRolePolicy(new iam.PolicyStatement({
      actions: ['sts:AssumeRole'],
      resources: [tahoeStage.apiInvocationRoleArn],
    }));

    onEventHandler.addToRolePolicy(new iam.PolicyStatement({
      actions: ['iam:PassRole'],
      resources: [`arn:aws:iam::${account}:role/*TapRolesData*`], // This should actually get the DataRole from TapRoles rather than try to infer it
    }))

    onEventHandler.addToRolePolicy(new iam.PolicyStatement({
      actions: ['s3:PutObject', 's3:PutObjectAcl'],
      resources: [`arn:aws:s3:::${tahoeStage.loggingBucketName}/*`]
    }))

    onEventHandler.addToRolePolicy(new iam.PolicyStatement({
      actions: ['s3:GetBucketAcl'],
      resources: [`arn:aws:s3:::${tahoeStage.loggingBucketName}`]
    }))

    onEventHandler.addToRolePolicy(new iam.PolicyStatement({
      actions: [
        'redshift:DescribeClusters',
        'redshift:ModifyClusterIamRoles',
        'redshift:EnableLogging',
        'ec2:DescribeAvailabilityZones',
      ],
      resources: ['*'],
    }));

    const { serviceToken } = new cr.Provider(this, 'Provider', {
      onEventHandler,
      logRetention: logs.RetentionDays.ONE_MONTH,
    });
    this.serviceToken = serviceToken
  }
}
