import * as iam from '@aws-cdk/aws-iam';
import * as sm from '@aws-cdk/aws-secretsmanager';
import * as cdk from '@aws-cdk/core';
import { TahoeProd, TahoeStage } from '.';

export interface TapRolesProps {
  tahoeStage?: TahoeStage;
  secret: sm.ISecret;
}

export class TapRoles extends cdk.Construct {
  public dataRole: iam.IRole;
  public adminRole: iam.IRole;

  constructor(scope: cdk.Construct, id: string, props: TapRolesProps) {
    super(scope, id);

    const account = cdk.Stack.of(this).account;
    const region = cdk.Stack.of(this).region;
    const tahoeAccount = (props.tahoeStage ?? TahoeProd).account;
    this.adminRole = new iam.Role(this, 'AdminRole', {
      assumedBy: new iam.CompositePrincipal(
        new iam.AccountRootPrincipal(),
        new iam.AccountPrincipal(tahoeAccount),
        new iam.ServicePrincipal('events.amazonaws.com'),
      ),
      // Tahoe API's lambda is only permitted to assume roles that match
      // 'arn:aws:iam::*:role/TapAdmin*', so we customize the role name to
      // match.
      roleName: `TapAdmin${this.node.addr}`,
    });
    const adminPolicy = new iam.Policy(this, 'AdminPolicy', {
      statements: [
        new iam.PolicyStatement({
          actions: [
            'cloudformation:DescribeStacks',
            'glue:*',
          ],
          resources: ['*'],
        }),
        new iam.PolicyStatement({
          actions: ['secretsmanager:GetSecretValue'],
          resources: [props.secret.secretArn],
        }),
      ]
    });
    // Event bridge policies to get status about local views.
    const tapEventBridgePolicy = new iam.Policy(this, 'TapEventBridgePolicy', {
      statements: [
        new iam.PolicyStatement({
          actions: [
            'events:PutRule',
            'events:DeleteRule',
            'events:DescribeRule',
            'events:PutTargets',
            'events:RemoveTargets',
            'events:PutEvents',
            'events:PutPermission',
            'events:TestEventPattern',
            'events:ListTargetsByRule',
          ],
          resources: ['*'],
        }),
        new iam.PolicyStatement({
          actions: ['events:PutEvents'],
          resources: [`arn:aws:events:us-west-2:${tahoeAccount}:event-bus/tahoe-local-views`],
        }),
        new iam.PolicyStatement({
          actions: ['iam:PassRole'],
          resources: ['arn:aws:iam::*:role/*'],
          conditions: {
            StringLike: {
              'iam:PassedToService': 'events.amazonaws.com'
            }
          }
        }),
      ]
    });
    const tapDataApiPolicy = new iam.Policy(this, 'TapRedshiftDataAPIPolicy', {
      statements: [
        new iam.PolicyStatement({
          actions: [
            'redshift-data:BatchExecuteStatement',
            'redshift-data:ExecuteStatement',
            'redshift-data:ListDatabases',
            'redshift-data:ListSchemas',
            'redshift-data:ListTables',
            'redshift-data:DescribeTable'
          ],
          resources: [`arn:aws:redshift:${region}:${account}:cluster:*`],
        }),
      ]
    });
    const tapDataApiStatementsPolicy = new iam.Policy(this, 'TapRedshiftDataAPIPolicyForStatements', {
      statements: [
        new iam.PolicyStatement({
          actions: [
            'redshift-data:CancelStatement',
            'redshift-data:GetStatementResult',
            'redshift-data:ListStatements',
            'redshift-data:DescribeStatement',
          ],
          resources: ['*'],
        }),
      ]
    });
    this.adminRole.attachInlinePolicy(adminPolicy);
    this.adminRole.attachInlinePolicy(tapEventBridgePolicy);
    this.adminRole.attachInlinePolicy(tapDataApiPolicy);
    this.adminRole.attachInlinePolicy(tapDataApiStatementsPolicy);

    this.dataRole = new iam.Role(this, 'DataRole', {
      assumedBy: new iam.ServicePrincipal('redshift.amazonaws.com'),
    });
    const dataAccessPolicy = new iam.Policy(this, 'DataAccessPolicy', {
      statements: [
        new iam.PolicyStatement({
          actions: [
            'glue:BatchGet*',
            'glue:Get*',
            'glue:List*',
            'kms:Decrypt',
            's3:Get*',
            's3:List*',
          ],
          resources: ['*']
        }),
      ]
    });
    this.dataRole.attachInlinePolicy(dataAccessPolicy);
  }
}
