import cdk = require('@aws-cdk/core');
import ecs = require('@aws-cdk/aws-ecs');
import ec2 = require('@aws-cdk/aws-ec2');
import iam = require('@aws-cdk/aws-iam');

interface ClusterProps {
  vpc: ec2.IVpc;
  dnsNamespace: string;
  environment?: { [key: string]: string };
}

export class Cluster extends cdk.Construct {
  public readonly zkNodes: string[];

  constructor(scope: cdk.Construct, id: string, props: ClusterProps) {
    super(scope, id);

    const cluster = new ecs.Cluster(this, 'Cluster', {
      vpc: props.vpc,
      defaultCloudMapNamespace: { name: props.dnsNamespace },
    });

    const asg = cluster.addCapacity('DefaultAutoScalingGroup', {
      instanceType: ec2.InstanceType.of(ec2.InstanceClass.M5, ec2.InstanceSize.LARGE),
      desiredCapacity: 3,
      maxCapacity: 9,
      minCapacity: 3,
      taskDrainTime: cdk.Duration.seconds(0),
    });
    asg.addUserData('yum install -y https://s3.amazonaws.com/ec2-downloads-windows/SSMAgent/latest/linux_amd64/amazon-ssm-agent.rpm');
    asg.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName("AmazonSSMManagedInstanceCore"));

    const zookeeperPorts = [
      2181, // Client port
      2888, // Election port
      3888, // Follower port
      8080, // Admin port
    ];

    this.zkNodes = [];
    [1, 2, 3].forEach(id => {
      this.zkNodes.push(`${id}.${props.dnsNamespace}`);

      const taskDefinition = new ecs.Ec2TaskDefinition(this, `Task${id}`, {
        networkMode: ecs.NetworkMode.AWS_VPC,
        volumes: [{
          name: 'data',
        }, {
          name: 'datalog',
        }],
      });

      const serverList = getServerList({
        namespace: props.dnsNamespace,
        count: 3,
        id,
      });

      const container = taskDefinition.addContainer("Container", {
        image: ecs.ContainerImage.fromRegistry('zookeeper:3.6.2'),
        memoryReservationMiB: 500,
        logging: new ecs.AwsLogDriver({
          logRetention: 30,
          streamPrefix: 'ZookeeperNode',
        }),
        environment: {
          ZOO_MY_ID: id.toString(),
          ZOO_SERVERS: serverList,
          ZOO_STANDALONE_ENABLED: 'false',
          ZOO_AUTOPURGE_PURGEINTERVAL: '1',
          ZOO_4LW_COMMANDS_WHITELIST: '*',
          ...(props.environment || {}),
        },
        healthCheck: {
          command: ['CMD-SHELL', 'wget -q -O - localhost:8080/commands/stat || exit 1'],
        },
      });


      container.addMountPoints({
        containerPath: '/data',
        sourceVolume: 'data',
        readOnly: false,
      }, {
        containerPath: '/datalog',
        sourceVolume: 'datalog',
        readOnly: false,
      });

      // Expose all Zookeeper ports
      container.addPortMappings(...zookeeperPorts.map(port => ({
        containerPort: port,
        hostPort: port,
      })));

      const service = new ecs.Ec2Service(this, `ZKNode${id}`, {
        cluster,
        taskDefinition,
        minHealthyPercent: 0,
        desiredCount: 1,
        cloudMapOptions: {
          dnsTtl: cdk.Duration.seconds(10),
          name: id.toString(),
        },
        placementConstraints: [ecs.PlacementConstraint.distinctInstances()],
      });

      // Make all zookeeper ports accessible on the service.
      zookeeperPorts.forEach(port => {
        service.connections.allowFromAnyIpv4(ec2.Port.tcp(port));
      });
    });
  }
}

interface GetServerListParams {
  namespace: string,
  count: number,
  id: number,
}

/**
 * Build a Zookeeper server list property. These string values change for each
 * node and follow this pattern:
 *
 * server.1=0.0.0.0:2888:3888;2181 server.2=2.zookeeper:2888:3888;2181 server.3=3.zookeeper:2888:3888;2181
 * server.1=1.zookeeper:2888:3888;2181 server.2=0.0.0.0:2888:3888;2181 server.3=3.zookeeper:2888:3888;2181
 * server.1=1.zookeeper:2888:3888;2181 server.2=2.zookeeper:2888:3888;2181 server.3=0.0.0.0:2888:3888;2181
 */
function getServerList(params: GetServerListParams) {
  const entries: string[] = [];
  let current = 1;
  while (current <= params.count) {
    const host = params.id === current ? '0.0.0.0' : `${current}.${params.namespace}`;
    const entry = `server.${current}=${host}:2888:3888;2181`;
    entries.push(entry);
    current++;
  }

  return entries.join(' ');
}
