package clients

import (
	"context"
	"time"

	friendship "code.justin.tv/amzn/TwitchFriendshipTwirp"
	"code.justin.tv/foundation/twitchclient"
)

const (
	friendshipIsFriendTimeout = 150 * time.Millisecond
)

// Friendship is an interface based on chat's friendship client
type Friendship interface {
	IsFriend(ctx context.Context, userID, targetID string, reqOpts *twitchclient.ReqOpts) (bool, error)
}

// FriendshipClient used within meepo
type FriendshipClient interface {
	IsFriend(ctx context.Context, sourceID, targetID string) (bool, error)
}

type friendshipImpl struct {
	baseClient friendship.Friends
}

// NewFriendshipClient creates a new client for use within meepo
func NewFriendshipClient(host string, stats twitchclient.Statter) (FriendshipClient, error) {
	httpClient := twitchclient.NewHTTPClient(twitchclient.ClientConf{
		Transport: twitchclient.TransportConf{
			MaxIdleConnsPerHost: 500,
		},
		Host:           host,
		TimingXactName: "friendship",
		Stats:          stats,
	})

	friendshipClient := friendship.NewFriendsProtobufClient(host, httpClient)

	return &friendshipImpl{baseClient: friendshipClient}, nil
}

func (u *friendshipImpl) IsFriend(ctx context.Context, sourceID, targetID string) (bool, error) {
	ctx, cancel := context.WithTimeout(ctx, friendshipIsFriendTimeout)
	defer cancel()

	isFriend, err := u.baseClient.IsFriend(ctx, &friendship.IsFriendReq{
		SourceUserId: sourceID,
		TargetUserId: targetID,
	})
	if err != nil {
		return false, err
	}
	return isFriend.GetIsFriend(), nil
}
