package clients

import (
	"context"

	"code.justin.tv/discovery/liveline/proto/liveline"
	"code.justin.tv/foundation/twitchclient"
)

const (
	livelineQueryBatchSize = 100
)

// Liveline is an interface based on discovery's Liveline client
type Liveline interface {
	GetFilteredStreamsByChannelIDs(context.Context, *liveline.FilteredStreamsByChannelIDsRequest) (*liveline.StreamsResponse, error)
}

// LivelineClient used within meepo
type LivelineClient interface {
	GetLiveChannelsByChannelIDs(ctx context.Context, channelIDs []string) ([]string, error)
}

type livelineImpl struct {
	baseClient Liveline
}

// NewLivelineClient creates a new client for use within meepo
func NewLivelineClient(host string, stats twitchclient.Statter) LivelineClient {
	httpClient := twitchclient.NewHTTPClient(twitchclient.ClientConf{
		Transport: twitchclient.TransportConf{
			MaxIdleConnsPerHost: 100,
		},
		Host:  host,
		Stats: stats,
	})

	livelineClient := liveline.NewLivelineProtobufClient(host, httpClient)

	return &livelineImpl{baseClient: livelineClient}
}

func (u *livelineImpl) GetLiveChannelsByChannelIDs(ctx context.Context, channelIDs []string) ([]string, error) {
	liveChannelIDs := make([]string, 0, len(channelIDs))
	for low := 0; low < len(channelIDs); low += livelineQueryBatchSize {
		high := low + livelineQueryBatchSize
		if high > len(channelIDs) {
			high = len(channelIDs)
		}
		localIDs := channelIDs[low:high]

		// get live channels and not reruns, playlists, etc
		filters := []*liveline.Filter{
			{
				FilterKey:   liveline.FilterKey_STREAM_TYPE,
				FilterValue: []string{"live"},
			},
		}

		req := &liveline.FilteredStreamsByChannelIDsRequest{
			ChannelIds: localIDs,
			Filters:    filters,
		}

		res, err := u.baseClient.GetFilteredStreamsByChannelIDs(ctx, req)
		if err != nil {
			return nil, err
		}

		for _, stream := range res.GetStreams() {
			if stream.GetChannelId() != "" {
				liveChannelIDs = append(liveChannelIDs, stream.GetChannelId())
			}
		}
	}

	return liveChannelIDs, nil
}
