package clients

import (
	"encoding/json"
	"time"

	pubsub "code.justin.tv/chat/pubsub-go-pubclient/client"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/foundation/twitchclient"
	"code.justin.tv/twitch-events/meepo/internal/models"

	"golang.org/x/net/context"
)

// Pubsub topic prefixes.
const (
	squadUpdatesByChannelIDTopicPrefix = "channel-squad-updates."
	squadUpdatesBySquadIDTopicPrefix   = "squad-updates."

	// TODO: Change this topic to something more descriptive, like "squad-manager-updates."
	creatorUpdatesByChannelIDTopicPrefix = "channel-squad-invites."
)

// Pubsub types for channel-squad-invites.
const (
	PubsubTypeReceivedSquadInvites = "received-squad-invites"
	PubsubTypeSquad                = "squad"
	PubsubTypeManagedSquad         = "managed-squad"
)

// PubSub is an interface based on chat's go-pubclient
type PubSub interface {
	Publish(context.Context, []string, string, *twitchclient.ReqOpts) error
}

// PubsubClient used within meepo
type PubsubClient interface {
	PublishSquadUpdate(ctx context.Context, managedSquad *models.PubsubManagedSquad) error
	PublishChannelNotInSquad(ctx context.Context, channelID string) error
	PublishIncomingSquadInvites(ctx context.Context, recipientID string, invitations []*models.PubsubIncomingInvitation) error
}

type pubsubImpl struct {
	baseClient PubSub
}

// NewPubSubClient creates a new client for use within meepo
func NewPubSubClient(host string, stats twitchclient.Statter) (PubsubClient, error) {
	clientConf := twitchclient.ClientConf{
		Transport: twitchclient.TransportConf{
			MaxIdleConnsPerHost: 100,
		},
		Host:  host,
		Stats: stats,
	}

	pubsubClient, err := pubsub.NewPubClient(clientConf)
	if err != nil {
		return nil, errors.New("failed to start pubsub client")
	}

	return &pubsubImpl{baseClient: pubsubClient}, nil
}

// SquadUpdate represents a squad update message
type SquadUpdate struct {
	Squad     *models.PubsubSquad `json:"squad"`
	Timestamp time.Time           `json:"timestamp"`
	Type      string              `json:"type"`
}

// ManagedSquadUpdate represents a squad update message meant for the owner of the squad.
type ManagedSquadUpdate struct {
	Squad     *models.PubsubManagedSquad `json:"squad"`
	Timestamp time.Time                  `json:"timestamp"`
	Type      string                     `json:"type"`
}

// PublishSquadUpdate sends information on the current squad to our various Pubsub topics.  It is meant to be
// used whenever the squad's status, members or outgoing invitations change.
//
// Updates need to go to three sets of Pubsub topics:
//   - The squad's public topic (indexed by squad ID).  This topic drives updates to the squad page.
//   - Each member's public topic (indexed by channel ID).  This topic drives updates to the banner on each member's
//   - channel page.
//   - Each member's private creator topic (indexed by channel ID.)  This topic drives updates to the creator's
//     dashboard widget.
//
// The public topics, and the private creator topic for members that do not own the squad, should get info on the squad
// that is public.
// The private creator topic for the owner of the squad needs more information that only owners (and other trusted
// users) are allowed to see.  They need this extra info to allow the owner to manage their squad's outgoing invitations.
func (p *pubsubImpl) PublishSquadUpdate(ctx context.Context, managedSquad *models.PubsubManagedSquad) error {
	if managedSquad == nil {
		return nil
	}

	err := p.publishManagedSquadUpdates(ctx, managedSquad)
	if err != nil {
		return err
	}

	return p.publishPublicSquadUpdates(ctx, &models.PubsubSquad{
		ID:      managedSquad.ID,
		Members: managedSquad.Members,
		OwnerID: managedSquad.OwnerID,
		Status:  managedSquad.Status,
	})
}

func (p *pubsubImpl) publishManagedSquadUpdates(ctx context.Context, squad *models.PubsubManagedSquad) error {
	if squad == nil || squad.OwnerID == nil || *squad.OwnerID == "" {
		return nil
	}
	ownerID := *squad.OwnerID
	topicForOwnerUpdates := []string{getCreatorUpdatesByChannelIDTopic(ownerID)}

	pubsubMessage, err := json.Marshal(ManagedSquadUpdate{
		Squad:     squad,
		Timestamp: time.Now().UTC(),
		Type:      PubsubTypeManagedSquad,
	})
	if err != nil {
		return errors.Wrapf(err, "sending owner squad update to pubsub failed. squad_id: %s", squad.ID)
	}

	err = p.baseClient.Publish(ctx, topicForOwnerUpdates, string(pubsubMessage), nil)
	if err != nil {
		return errors.Wrapf(err, "sending owner squad update to pubsub failed. squad_id: %s", squad.ID)
	}

	return nil
}

func (p *pubsubImpl) publishPublicSquadUpdates(ctx context.Context, squad *models.PubsubSquad) error {
	if squad == nil {
		return nil
	}

	var ownerID string
	if squad.OwnerID != nil {
		ownerID = *squad.OwnerID
	}

	// We will need to publish this squad update to:
	//   - The squad's public topic (that powers the squad page)
	//   - Each member's public topic (that powers the banner on each members channel page)
	//   - Each non-owner member's private "creator" topic (that drives updates to the widget)
	//     We don't send a public squad update to the owner's creator topic because we send a managed-squad update
	//     that contains more information to them.

	topics := make([]string, 0, len(squad.Members)*2) // capacity is approx: 1 sq topic + num_members public topics + num_non_members private topics
	topics = append(topics, getSquadUpdatesBySquadIDTopic(squad.ID))

	for _, member := range squad.Members {
		topics = append(topics, getSquadUpdatesByChannelIDTopic(member.ID))
	}

	for _, member := range squad.Members {
		if member.ID == ownerID {
			continue
		}
		topics = append(topics, getCreatorUpdatesByChannelIDTopic(member.ID))
	}

	pubsubMessage, err := json.Marshal(SquadUpdate{
		Squad:     squad,
		Timestamp: time.Now().UTC(),
		Type:      PubsubTypeSquad,
	})
	if err != nil {
		return errors.Wrapf(err, "sending public squad update to pubsub failed. squad_id: %s", squad.ID)
	}

	err = p.baseClient.Publish(ctx, topics, string(pubsubMessage), nil)
	if err != nil {
		return errors.Wrapf(err, "sending public squad update to pubsub failed. squad_id: %s", squad.ID)
	}

	return nil
}

// PublishChannelNotInSquad is used to publish that a channel is no longer part of a squad.
// We publish to the channel's public and private topics to remove the banner, and to clear the Dashboard widget.
func (p *pubsubImpl) PublishChannelNotInSquad(ctx context.Context, channelID string) error {
	topics := []string{
		getSquadUpdatesByChannelIDTopic(channelID),
		getCreatorUpdatesByChannelIDTopic(channelID),
	}

	update := SquadUpdate{
		Squad:     nil,
		Timestamp: time.Now().UTC(),
		Type:      PubsubTypeSquad,
	}

	pubsubMessage, err := json.Marshal(update)
	if err != nil {
		return errors.Wrap(err, "sending 'members not in squad' update to pubsub failed")
	}

	err = p.baseClient.Publish(ctx, topics, string(pubsubMessage), nil)
	if err != nil {
		return errors.Wrap(err, "sending 'members not in squad' update to pubsub failed")
	}

	return nil
}

// IncomingInvitesUpdate represents a squad invites update message
type IncomingInvitesUpdate struct {
	Invitations []*models.PubsubIncomingInvitation `json:"invitations"`
	Timestamp   time.Time                          `json:"timestamp"`
	Type        string                             `json:"type"`
}

// PublishIncomingSquadInvites sends the list of a channel's incoming invitations to the channel's private "creator"
// topic.  It is meant to trigger updates to the Invites tab of the Dashboard widget.
func (p *pubsubImpl) PublishIncomingSquadInvites(ctx context.Context, recipientID string, invitations []*models.PubsubIncomingInvitation) error {
	topics := []string{getCreatorUpdatesByChannelIDTopic(recipientID)}
	update := IncomingInvitesUpdate{
		Invitations: invitations,
		Timestamp:   time.Now().UTC(),
		Type:        PubsubTypeReceivedSquadInvites,
	}

	pubsubMessage, err := json.Marshal(update)
	if err != nil {
		return errors.Wrapf(err, "sending incoming invites update to pubsub failed. recipient_id: %s", recipientID)
	}

	err = p.baseClient.Publish(ctx, topics, string(pubsubMessage), nil)
	if err != nil {
		return errors.Wrapf(err, "sending incoming invites update to pubsub failed. recipient_id: %s", recipientID)
	}

	return nil
}

func getSquadUpdatesByChannelIDTopic(channelID string) string {
	return squadUpdatesByChannelIDTopicPrefix + channelID
}

func getSquadUpdatesBySquadIDTopic(squadID string) string {
	return squadUpdatesBySquadIDTopicPrefix + squadID
}

func getCreatorUpdatesByChannelIDTopic(channelID string) string {
	return creatorUpdatesByChannelIDTopicPrefix + channelID
}
