package clients

import (
	"context"
	"time"

	api "code.justin.tv/cb/roster/api/v1"
	"code.justin.tv/cb/roster/client/roster"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/foundation/twitchclient"
)

const (
	rosterGetV1ChannelMembershipsTimeout = 80 * time.Millisecond
	rosterGetV1TeamMembershipsTimeout    = 30 * time.Millisecond
)

// Roster is an interface based on cb's roster client
type Roster interface {
	GetV1ChannelMemberships(ctx context.Context, params api.GetChannelMembershipsRequestParams, opts *twitchclient.ReqOpts) (*api.GetChannelMembershipsResponse, error)
	GetV1TeamMemberships(ctx context.Context, params api.GetTeamMembershipsRequestParams, opts *twitchclient.ReqOpts) (*api.GetTeamMembershipsResponse, error)
}

// RosterClient used within meepo
type RosterClient interface {
	IsTeammate(ctx context.Context, sourceID, targetID string) (bool, error)
}

type rosterImpl struct {
	baseClient Roster
}

// NewRosterClient creates a new client for use within meepo
func NewRosterClient(host string, stats twitchclient.Statter) (RosterClient, error) {
	clientConf := twitchclient.ClientConf{
		Transport: twitchclient.TransportConf{
			MaxIdleConnsPerHost: 100,
		},
		Host:  host,
		Stats: stats,
	}

	rosterClient, err := roster.NewClient(clientConf)
	if err != nil {
		return nil, errors.New("failed to start roster client")
	}

	return &rosterImpl{baseClient: rosterClient}, nil
}

func (u *rosterImpl) IsTeammate(ctx context.Context, sourceID, targetID string) (bool, error) {
	ctx1, cancel1 := context.WithTimeout(ctx, rosterGetV1ChannelMembershipsTimeout)
	defer cancel1()

	channelRes, err := u.baseClient.GetV1ChannelMemberships(ctx1, api.GetChannelMembershipsRequestParams{
		ChannelID: sourceID,
	}, nil)
	if err != nil {
		return false, err
	}
	for _, teamData := range channelRes.Data {
		if !teamData.Primary {
			continue
		}

		ctx2, cancel2 := context.WithTimeout(ctx, rosterGetV1TeamMembershipsTimeout)
		defer cancel2()

		teamRes, err := u.baseClient.GetV1TeamMemberships(ctx2, api.GetTeamMembershipsRequestParams{
			TeamID: teamData.Team.ID,
		}, nil)
		if err != nil {
			return false, err
		}

		for _, membership := range teamRes.Data {
			if targetID == membership.ChannelID {
				return true, nil
			}
		}
		break
	}
	return false, nil
}
