package main

import (
	"context"
	"log"
	"os"

	"github.com/aws/aws-lambda-go/lambda"

	eventbus "code.justin.tv/eventbus/client"
	"code.justin.tv/eventbus/client/subscriber/lambdafunc"
	"code.justin.tv/eventbus/schema/pkg/user"
	"code.justin.tv/eventbus/schema/pkg/user_ban"
	"code.justin.tv/foundation/twitchclient"
	"code.justin.tv/twitch-events/meepo/rpc/meepo"
)

type handler struct {
	MeepoClient meepo.InternalMeepo
}

func main() {
	meepoHost := os.Getenv("MEEPO_HOST")
	if meepoHost == "" {
		log.Fatalf("meepo host not found")
	}

	clientConf := twitchclient.ClientConf{
		Transport: twitchclient.TransportConf{
			MaxIdleConnsPerHost: 100,
		},
		Host: meepoHost,
	}
	meepoClient := meepo.NewInternalMeepoProtobufClient(meepoHost, twitchclient.NewHTTPClient(clientConf))
	h := &handler{
		MeepoClient: meepoClient,
	}

	mux := eventbus.NewMux()
	user.RegisterDestroyHandler(mux, h.UserDestroyed)
	user.RegisterDeleteHandler(mux, h.UserDeleted)
	user_ban.RegisterUserBanCreateHandler(mux, h.UserBanned)

	lambda.Start(lambdafunc.NewSQS(mux.Dispatcher()))
}

func (h *handler) UserDestroyed(ctx context.Context, header *eventbus.Header, event *user.Destroy) error {
	if event == nil {
		log.Printf("UserDestroyed: nil eventbus event")
		return nil
	}

	userID := event.GetUserId()
	if userID == "" {
		log.Printf("UserDestroyed: userID cannot be empty")
		return nil
	}

	_, err := h.MeepoClient.DestroyUser(ctx, &meepo.DestroyUserRequest{
		ChannelId: userID,
	})

	if err != nil {
		log.Printf("UserDestroyed: failed to destroy user %s from Meepo, err = %v", userID, err)
		return err
	}

	log.Printf("UserDestroyed: user %s destroyed for compliance", userID)
	return nil
}

func (h *handler) UserDeleted(ctx context.Context, header *eventbus.Header, event *user.Delete) error {
	if event == nil {
		log.Printf("UserDeleted: nil eventbus event")
		return nil
	}

	userID := event.GetUserId()
	if userID == "" {
		log.Printf("UserDeleted: userID cannot be empty")
		return nil
	}

	_, err := h.MeepoClient.DeleteUser(ctx, &meepo.DeleteUserRequest{
		ChannelId: userID,
	})

	if err != nil {
		log.Printf("UserDeleted: failed to delete user %s from Meepo, err = %v", userID, err)
		return err
	}

	log.Printf("UserDeleted: user %s deleted for compliance", userID)
	return nil
}

func (h *handler) UserBanned(ctx context.Context, header *eventbus.Header, event *user_ban.UserBanCreate) error {
	if event == nil {
		log.Printf("UserBanned: nil eventbus event")
		return nil
	}

	userID := event.GetUserId()
	if userID == "" {
		log.Printf("UserBanned: userID cannot be empty")
		return nil
	}

	_, err := h.MeepoClient.DeleteUser(ctx, &meepo.DeleteUserRequest{
		ChannelId: userID,
	})

	if err != nil {
		log.Printf("UserBanned: failed to ban user %s from Meepo, err = %v", userID, err)
		return err
	}

	log.Printf("UserBanned: user %s banned for compliance", userID)
	return nil
}
