resource "aws_iam_role" "channel_livecheck_lambda_role" {
  name = "${var.name}-${var.environment}-${var.lambda_name}-role"

  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": "sts:AssumeRole",
      "Effect": "Allow",
      "Principal": {
        "Service": "lambda.amazonaws.com"
      },
      "Sid": ""
    }
  ]
}
EOF
}

resource "aws_cloudwatch_log_group" "channel_livecheck_lambda_log_group" {
  name = "/aws/lambda/${var.name}-${var.environment}-${var.lambda_name}"
}

resource "aws_iam_role_policy_attachment" "lambda_policy_attach" {
  role       = aws_iam_role.channel_livecheck_lambda_role.name
  policy_arn = "arn:aws:iam::aws:policy/service-role/AWSLambdaVPCAccessExecutionRole"
}

resource "aws_iam_role_policy" "channel_livecheck_lambda_basic_execution" {
  name = "${var.name}-${var.environment}-channel-livecheck-lambda-basic-execution"
  role = aws_iam_role.channel_livecheck_lambda_role.id

  policy = <<EOF
{
    "Version": "2012-10-17",
    "Statement": [
        {
            "Effect": "Allow",
            "Action": [
                "logs:CreateLogStream",
                "logs:PutLogEvents"
            ],
            "Resource": [
                "${aws_cloudwatch_log_group.channel_livecheck_lambda_log_group.arn}"
            ]
        }
    ]
}
EOF
}

// Zips up Lambda function folder
data "archive_file" "channel_livecheck_lambda_zip" {
  type        = "zip"
  source_dir  = var.lambda_directory
  output_path = "archive/${var.lambda_zip_filename}"
}

resource "aws_lambda_function" "channel_livecheck_lambda" {
  filename         = "archive/${var.lambda_zip_filename}"
  function_name    = "${var.name}-${var.environment}-${var.lambda_name}"
  role             = aws_iam_role.channel_livecheck_lambda_role.arn
  handler          = "main"
  source_code_hash = data.archive_file.channel_livecheck_lambda_zip.output_base64sha256
  runtime          = "go1.x"
  description      = "${var.team_display_name} team's lambda function to check if channels are live in ${var.service_name} service for ${var.environment}"
  timeout          = 60

  environment {
    variables = {
      MEEPO_HOST = var.meepo_host
    }
  }

  vpc_config {
    subnet_ids         = var.private_subnets
    security_group_ids = var.twitch_subnets_sg
  }

  tags = {
    Owner   = var.owner
    Project = var.project
    Team    = var.team_display_name
  }
}

/* Taken from: https://github.com/hashicorp/terraform/issues/4393#issuecomment-194287540*/

resource "aws_cloudwatch_event_rule" "channel_livecheck_event" {
  count               = var.is_scheduled ? 1 : 0
  name                = "${var.name}-${var.environment}-${var.lambda_name}-event"
  description         = "Periodically triggers lambda to check if channels are live"
  schedule_expression = var.rate
}

resource "aws_cloudwatch_event_target" "channel_livecheck_event_target" {
  count     = var.is_scheduled ? 1 : 0
  rule      = aws_cloudwatch_event_rule.channel_livecheck_event[0].name
  target_id = aws_lambda_function.channel_livecheck_lambda.function_name
  arn       = aws_lambda_function.channel_livecheck_lambda.arn
}

resource "aws_lambda_permission" "channel_livecheck_event_lambda_permission" {
  count         = var.is_scheduled ? 1 : 0
  statement_id  = "AllowExecutionFromCloudWatch"
  action        = "lambda:InvokeFunction"
  function_name = aws_lambda_function.channel_livecheck_lambda.function_name
  principal     = "events.amazonaws.com"
  source_arn    = aws_cloudwatch_event_rule.channel_livecheck_event[0].arn
}

data "aws_sns_topic" "pagerduty_sns_topic" {
  count = var.is_scheduled ? 1 : 0
  name  = "pagerduty_${var.service_name}"
}

resource "aws_cloudwatch_metric_alarm" "channel_livecheck_lambda_invocations" {
  count      = var.is_scheduled ? 1 : 0
  alarm_name = "${aws_lambda_function.channel_livecheck_lambda.function_name}-invocations"

  comparison_operator = "LessThanOrEqualToThreshold"
  evaluation_periods  = 2
  metric_name         = "Invocations"
  namespace           = "AWS/Lambda"
  period              = 300
  statistic           = "Sum"
  threshold           = 0

  dimensions = {
    FunctionName = aws_lambda_function.channel_livecheck_lambda.function_name
  }

  alarm_actions      = [data.aws_sns_topic.pagerduty_sns_topic[0].arn]
  ok_actions         = [data.aws_sns_topic.pagerduty_sns_topic[0].arn]
  treat_missing_data = "notBreaching"
}

resource "aws_cloudwatch_metric_alarm" "channel_livecheck_lambda_errors" {
  count      = var.is_scheduled ? 1 : 0
  alarm_name = "${aws_lambda_function.channel_livecheck_lambda.function_name}-errors"

  comparison_operator = "GreaterThanOrEqualToThreshold"
  evaluation_periods  = 1
  metric_name         = "Errors"
  namespace           = "AWS/Lambda"
  period              = 300
  statistic           = "Sum"
  threshold           = 1

  dimensions = {
    FunctionName = aws_lambda_function.channel_livecheck_lambda.function_name
  }

  alarm_actions      = [data.aws_sns_topic.pagerduty_sns_topic[0].arn]
  ok_actions         = [data.aws_sns_topic.pagerduty_sns_topic[0].arn]
  treat_missing_data = "notBreaching"
}
