resource "aws_iam_role" "eventbus_lambda_role" {
  name = "${var.name}-${var.environment}-${var.lambda_name}-role"

  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": "sts:AssumeRole",
      "Effect": "Allow",
      "Principal": {
        "Service": "lambda.amazonaws.com"
      },
      "Sid": ""
    }
  ]
}
EOF
}

resource "aws_cloudwatch_log_group" "eventbus_lambda_log_group" {
  name = "/aws/lambda/${var.name}-${var.environment}-${var.lambda_name}"
}

resource "aws_iam_role_policy_attachment" "eventbus_lambda_vpc_policy_attach" {
  role       = aws_iam_role.eventbus_lambda_role.name
  policy_arn = "arn:aws:iam::aws:policy/service-role/AWSLambdaVPCAccessExecutionRole"
}

resource "aws_iam_role_policy" "eventbus_lambda_basic_execution" {
  name = "${var.name}-${var.environment}-eventbus-lambda-basic-execution"
  role = aws_iam_role.eventbus_lambda_role.id

  policy = <<EOF
{
    "Version": "2012-10-17",
    "Statement": [
        {
            "Effect": "Allow",
            "Action": [
                "logs:CreateLogStream",
                "logs:PutLogEvents"
            ],
            "Resource": [
                "${aws_cloudwatch_log_group.eventbus_lambda_log_group.arn}"
            ]
        }
    ]
}
EOF
}

// Zips up Lambda function folder
data "archive_file" "eventbus_lambda_zip" {
  type        = "zip"
  source_dir  = var.lambda_directory
  output_path = "archive/${var.lambda_zip_filename}"
}

resource "aws_lambda_function" "eventbus_lambda" {
  filename         = "archive/${var.lambda_zip_filename}"
  function_name    = "${var.name}-${var.environment}-${var.lambda_name}"
  role             = aws_iam_role.eventbus_lambda_role.arn
  handler          = "main"
  source_code_hash = data.archive_file.eventbus_lambda_zip.output_base64sha256
  runtime          = "go1.x"
  description      = "${var.team_display_name} team's lambda function to delete data from ${var.service_name} service for ${var.environment} when a user is destroyed, soft-deleted, or banned"
  timeout          = 60

  environment {
    variables = {
      MEEPO_HOST = var.meepo_host
    }
  }

  vpc_config {
    subnet_ids         = var.private_subnets
    security_group_ids = var.twitch_subnets_sg
  }

  tags = {
    Owner   = var.owner
    Project = var.project
    Team    = var.team_display_name
  }
}

data "aws_cloudformation_stack" "eventbus" {
  name = "EventBus"
}

resource "aws_iam_role_policy_attachment" "eventbus_lambda_access_policy_attach" {
  role       = aws_iam_role.eventbus_lambda_role.name
  policy_arn = data.aws_cloudformation_stack.eventbus.outputs["EventBusAccessPolicyARN"]
}

resource "aws_sqs_queue" "eventbus_sqs_queue" {
  name = "${var.name}-${var.environment}-${var.lambda_name}"

  visibility_timeout_seconds = 60
  message_retention_seconds  = 604800 # 7 days

  policy            = data.aws_cloudformation_stack.eventbus.outputs["EventBusSQSPolicyDocument"]
  kms_master_key_id = data.aws_cloudformation_stack.eventbus.outputs["EventBusKMSMasterKeyARN"]

  redrive_policy = <<JSON
{
  "deadLetterTargetArn": "${aws_sqs_queue.eventbus_dead_letter.arn}",
  "maxReceiveCount": 5
}
JSON

  lifecycle {
    prevent_destroy = true
  }

  tags = {
    Owner   = var.owner
    Project = var.project
    Team    = var.team_display_name
  }
}

resource "aws_sqs_queue" "eventbus_dead_letter" {
  name                      = "${var.name}-${var.environment}-${var.lambda_name}-dead-letter"
  kms_master_key_id         = "${data.aws_cloudformation_stack.eventbus.outputs["EventBusKMSMasterKeyARN"]}"
  message_retention_seconds = 604800 # 7 days

  lifecycle {
    prevent_destroy = true
  }

  tags = {
    Owner   = var.owner
    Project = var.project
    Team    = var.team_display_name
  }
}

resource "aws_lambda_event_source_mapping" "sqs" {
  batch_size       = 1
  event_source_arn = aws_sqs_queue.eventbus_sqs_queue.arn
  function_name    = aws_lambda_function.eventbus_lambda.function_name
}

resource "aws_lambda_permission" "sqs" {
  statement_id  = "allow_sqs_to_invoke"
  action        = "lambda:InvokeFunction"
  function_name = aws_lambda_function.eventbus_lambda.function_name
  principal     = "sqs.amazonaws.com"
  source_arn    = aws_sqs_queue.eventbus_sqs_queue.arn
}

data "aws_sns_topic" "pagerduty_sns_topic" {
  count = var.environment == "production" ? 1 : 0
  name  = "pagerduty_${var.service_name}"
}

resource "aws_cloudwatch_metric_alarm" "eventbus_dead_letter_messages" {
  count      = var.environment == "production" ? 1 : 0
  alarm_name = "${aws_lambda_function.eventbus_lambda.function_name}-dead-letter-messages"

  comparison_operator = "GreaterThanOrEqualToThreshold"
  evaluation_periods  = 1
  metric_name         = "ApproximateNumberOfMessagesVisible"
  namespace           = "AWS/SQS"
  period              = 3600 # 1 hour
  statistic           = "Average"
  threshold           = 1

  dimensions = {
    QueueName = aws_sqs_queue.eventbus_dead_letter.name
  }

  alarm_actions      = [data.aws_sns_topic.pagerduty_sns_topic[0].arn]
  ok_actions         = [data.aws_sns_topic.pagerduty_sns_topic[0].arn]
  treat_missing_data = "notBreaching"
}
