terraform {
  required_version = ">= 0.12"

  backend "s3" {
    bucket  = "twitch-feed-aws"
    key     = "tfstate/events/meepo/terraform/production"
    region  = "us-west-2"
    profile = "twitch-feed-aws"
  }
}

# Create a data source to retrieve information on the VPC that Systems set up in the
# AWS account.
data "terraform_remote_state" "account" {
  backend = "s3"

  config = {
    bucket  = var.account
    key     = "tfstate/feed/terraform/accounts/${var.account}"
    region  = var.region
    profile = var.account
  }
}

module "meepo" {
  name              = var.name
  service_name      = var.service_name
  project           = var.project
  team_display_name = var.team_display_name
  team_name         = var.team_name
  owner             = var.owner

  source                       = "../../modules/meepo"
  instance_type                = "c5.9xlarge"
  asg_max_size                 = 24
  asg_min_size                 = 12
  environment                  = var.environment
  account                      = var.account
  aws_key_name                 = "events-meepo-production"
  sandstorm_role_arn           = "arn:aws:iam::734326455073:role/sandstorm/production/templated/role/events-production"

  vpc_id            = data.terraform_remote_state.account.outputs.vpc_id
  private_subnets   = split(",", data.terraform_remote_state.account.outputs.private_subnets)
  twitch_subnets_sg = split(",", data.terraform_remote_state.account.outputs.twitch_subnets_sg)

  elasticache_node_type  = "cache.r3.2xlarge"
  elasticache_node_count = "3"
}

module "pagerduty" {
  source = "../../modules/cloudwatch"

  service_name                        = var.service_name
  environment                         = var.environment
  elastic_beanstalk_environment_name  = module.meepo.environment_name
  elastic_beanstalk_asg_name          = module.meepo.asg_name
  elastic_beanstalk_loadbalancer_name = module.meepo.elb_name
  pagerduty_endpoint                  = "https://events.pagerduty.com/integration/01110ef899574056ab741ccc8ef39bea/enqueue"
  elasticache_cluster_id              = module.meepo.elasticache_cluster_id
}

module "route53" {
  source = "../../modules/route53"

  service_name      = var.service_name
  project           = var.project
  team_display_name = var.team_display_name
  owner             = var.owner
  environment       = var.environment


  elb_dns_name = module.meepo.elb_dns_name
  elb_zone_id  = module.meepo.elb_zone_id
}

module "vpc_endpoint_service" {
  source = "../../modules/vpc_endpoint_service"

  account_id        = var.account_id
  name              = var.name
  service_name      = var.service_name
  project           = var.project
  team_display_name = var.team_display_name
  owner             = var.owner
  environment       = var.environment

  elb_dns_name    = module.meepo.elb_dns_name
  vpc_id          = data.terraform_remote_state.account.outputs.vpc_id
  private_subnets = split(",", data.terraform_remote_state.account.outputs.private_subnets)
  allowed_principals = [
    "arn:aws:iam::635065150292:root", // GrandDAds
  ]
}

output "elb_dns_name" {
  value = module.meepo.elb_dns_name
}

output "elb_zone_id" {
  value = module.meepo.elb_zone_id
}
