package log

import (
	"sync"
	"time"
)

// A RateLimitedLogger will disable itself after a Limit number of log messages in a period.
// This can be useful if you want to prevent your logger from killing the thing it is logging to
// such as disk or network.
type RateLimitedLogger struct {
	// Buffer size of logs that can overflow.  Allows burst
	EventMaxBuffer int64

	// Allows a log once per time.Duration
	LogRate time.Duration

	Logger      Logger
	LimitLogger Logger
	Now         func() time.Time

	mu               sync.Mutex
	bufferUsed       int64
	nextTimeAddEvent time.Time
}

func (r *RateLimitedLogger) now() time.Time {
	if r.Now == nil {
		return time.Now()
	}
	return r.Now()
}

// newEvent pretends a new event happened and sees if a log should happen
func (r *RateLimitedLogger) newEvent(incAmount int64) bool {
	now := r.now()
	r.mu.Lock()
	defer r.mu.Unlock()
	if now.Before(r.nextTimeAddEvent) {
		if r.bufferUsed == r.EventMaxBuffer {
			return false
		}
		r.bufferUsed += incAmount
		return true
	}
	gap := now.Sub(r.nextTimeAddEvent)
	eventsGivenBack := gap.Nanoseconds() / r.LogRate.Nanoseconds()
	r.bufferUsed -= eventsGivenBack
	if r.bufferUsed < 0 {
		r.bufferUsed = 0
	}
	r.bufferUsed += incAmount
	r.nextTimeAddEvent.Add(time.Duration(gap.Nanoseconds() * (1 + eventsGivenBack)))
	return true
}

// Log kvs to the wrapped Logger if the limit hasn't been reached
func (r *RateLimitedLogger) Log(kvs ...interface{}) {
	if r.newEvent(1) {
		if r.LimitLogger != nil {
			// Note: Log here messes up "caller" :/
			r.LimitLogger.Log(kvs...)
		}
		return
	}
	r.Logger.Log(kvs...)
}

// Disabled returns true if this logger is over its limit or if the wrapped logger is
// disabled.
func (r *RateLimitedLogger) Disabled() bool {
	return r.newEvent(0) || IsDisabled(r.Logger)
}
