package service_common

import (
	"fmt"
	"net/http"

	"io"
	"sync"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/log"
)

type panicErr struct {
	p interface{}
}

func (p *panicErr) Error() string {
	return fmt.Sprintf("panic: %v", p.p)
}

type PanicLogger interface {
	OnPanic(pnc interface{})
}

// HTTPLoggerMiddleware logs debug logs for each http path and regular logs for panics
type HTTPLoggerMiddleware struct {
	Log         *log.ElevatedLog
	PanicLogger PanicLogger
	CtxDim      *log.CtxDimensions
}

type PanicDumpWriter struct {
	Output io.Writer
	FlagOn *distconf.Bool
	mu     sync.Mutex
}

func (p *PanicDumpWriter) OnPanic(pnc interface{}) {
	p.mu.Lock()
	defer p.mu.Unlock()
	if !p.FlagOn.Get() {
		return
	}
	// Reuse the errors package to get a pretty-ish stack trace.
	err := errors.Wrap(&panicErr{pnc}, "panic")
	if stackErr, ok := err.(stackTraceErr); ok {
		fmt.Fprintf(p.Output, "%+v\n", stackErr.StackTrace())
	}
}

type stackTraceErr interface {
	StackTrace() errors.StackTrace
}

// NewHandler returns an http.Handler for request logging
func (m *HTTPLoggerMiddleware) NewHandler(h http.Handler) http.Handler {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		r = r.WithContext(m.CtxDim.Append(r.Context(), "in_http_method", r.Method, "in_http_path", r.URL.Path))
		defer func() {
			if p := recover(); p != nil {
				err := errors.Wrap(&panicErr{p}, "panic")
				m.Log.LogCtx(r.Context(), "err", err, "request panic!")
				if m.PanicLogger != nil {
					m.PanicLogger.OnPanic(p)
				}

				w.WriteHeader(http.StatusInternalServerError)
			}
		}()
		h.ServeHTTP(w, r)
	})
}
