// +build go1.8

package xray

import (
	"context"
	"database/sql"
)

// Begin wraps a SQL operation in an xray transaction
func (db *DB) Begin(ctx context.Context, opts *sql.TxOptions) (*Tx, error) {
	tx, err := db.db.BeginTx(ctx, opts)
	return &Tx{db, tx}, err
}

// Prepare wraps a SQL operation in an xray statement
func (db *DB) Prepare(ctx context.Context, query string) (*Stmt, error) {
	stmt, err := db.db.PrepareContext(ctx, enhanceQuery(ctx, query))
	return &Stmt{db, stmt, query}, err
}

// Ping captures to xray the DB ping
func (db *DB) Ping(ctx context.Context) error {
	return db.x.Capture(ctx, db.dbname, func(ctx context.Context) error {
		db.populate(ctx, "PING")
		return db.db.PingContext(ctx)
	})
}

// Exec captures to xray the DB Exec, including the full query
func (db *DB) Exec(ctx context.Context, query string, args ...interface{}) (sql.Result, error) {
	var res sql.Result

	err := db.x.Capture(ctx, db.dbname, func(ctx context.Context) error {
		db.populate(ctx, query)

		var err error
		res, err = db.db.ExecContext(ctx, enhanceQuery(ctx, query), args...)
		return err
	})

	return res, err
}

// Query captures to xray the DB Query, including the full query string
func (db *DB) Query(ctx context.Context, query string, args ...interface{}) (*sql.Rows, error) {
	var res *sql.Rows

	err := db.x.Capture(ctx, db.dbname, func(ctx context.Context) error {
		db.populate(ctx, query)

		var err error
		res, err = db.db.QueryContext(ctx, enhanceQuery(ctx, query), args...)
		return err
	})

	return res, err
}

// QueryRow captures to xray the DB QueryRow operation
func (db *DB) QueryRow(ctx context.Context, query string, args ...interface{}) *sql.Row {
	var res *sql.Row

	if err := db.x.Capture(ctx, db.dbname, func(ctx context.Context) error {
		db.populate(ctx, query)

		res = db.db.QueryRowContext(ctx, enhanceQuery(ctx, query), args...)
		return nil
	}); err != nil {
		db.x.onError("unexpected error from sql capture", err)
	}

	return res
}

// Exec captures to xray the transaction Exec timings
func (tx *Tx) Exec(ctx context.Context, query string, args ...interface{}) (sql.Result, error) {
	var res sql.Result

	err := tx.db.x.Capture(ctx, tx.db.dbname, func(ctx context.Context) error {
		tx.db.populate(ctx, query)

		var err error
		res, err = tx.tx.ExecContext(ctx, enhanceQuery(ctx, query), args...)
		return err
	})

	return res, err
}

// Query captures to xray the transaction Query timings
func (tx *Tx) Query(ctx context.Context, query string, args ...interface{}) (*sql.Rows, error) {
	var res *sql.Rows

	err := tx.db.x.Capture(ctx, tx.db.dbname, func(ctx context.Context) error {
		tx.db.populate(ctx, query)

		var err error
		res, err = tx.tx.QueryContext(ctx, enhanceQuery(ctx, query), args...)
		return err
	})

	return res, err
}

// QueryRow captures to xray the transaction QueryRow timings
func (tx *Tx) QueryRow(ctx context.Context, query string, args ...interface{}) *sql.Row {
	var res *sql.Row

	if err := tx.db.x.Capture(ctx, tx.db.dbname, func(ctx context.Context) error {
		tx.db.populate(ctx, query)

		res = tx.tx.QueryRowContext(ctx, enhanceQuery(ctx, query), args...)
		return nil
	}); err != nil {
		tx.db.x.onError("unexpected error from sql capture", err)
	}

	return res
}

// Stmt returns an xray SQL statement
func (tx *Tx) Stmt(ctx context.Context, stmt *Stmt) *Stmt {
	return &Stmt{stmt.db, tx.tx.StmtContext(ctx, stmt.stmt), stmt.query}
}

// Exec captures to xray the statement Exec timings
func (stmt *Stmt) Exec(ctx context.Context, args ...interface{}) (sql.Result, error) {
	var res sql.Result

	err := stmt.db.x.Capture(ctx, stmt.db.dbname, func(ctx context.Context) error {
		stmt.populate(ctx, stmt.query)

		var err error
		res, err = stmt.stmt.ExecContext(ctx, args...)
		return err
	})

	return res, err
}

// Query captures to xray the statement Query timings
func (stmt *Stmt) Query(ctx context.Context, args ...interface{}) (*sql.Rows, error) {
	var res *sql.Rows

	err := stmt.db.x.Capture(ctx, stmt.db.dbname, func(ctx context.Context) error {
		stmt.populate(ctx, stmt.query)

		var err error
		res, err = stmt.stmt.QueryContext(ctx, args...)
		return err
	})

	return res, err
}

// QueryRow captures to xray the statement QueryRow timings
func (stmt *Stmt) QueryRow(ctx context.Context, args ...interface{}) *sql.Row {
	var res *sql.Row

	if err := stmt.db.x.Capture(ctx, stmt.db.dbname, func(ctx context.Context) error {
		stmt.populate(ctx, stmt.query)

		res = stmt.stmt.QueryRowContext(ctx, args...)
		return nil
	}); err != nil {
		stmt.db.x.onError("unexpected error from sql capture", err)
	}

	return res
}
