package paramstoreconf

import (
	"context"
	"strings"

	"code.justin.tv/hygienic/distconf"
	"github.com/aws/aws-sdk-go/aws/request"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/ssm"
)

// Circuit allows callers to execute SSM calls inside a breaker that can support fast fails or timings
type Circuit interface {
	// Run should execute runFunc using the provided context
	Run(ctx context.Context, runFunc func(context.Context) error) error
}

type defaultCircuit struct{}

func (d defaultCircuit) Run(ctx context.Context, runFunc func(context.Context) error) error {
	return runFunc(ctx)
}

// ParameterStoreConfiguration allows loading configuration, possibly even secrets, from AWS's parameter store
type ParameterStoreConfiguration struct {
	Prefix         string
	Team           string
	Service        string
	Environment    string
	Circuit        Circuit
	AllowEncrypted bool
	SSM            *ssm.SSM
	distconf.ReaderCache
}

var _ distconf.Reader = &ParameterStoreConfiguration{}
var _ distconf.Dynamic = &ParameterStoreConfiguration{}

// DefaultParameterStoreConfiguration is a simple helper for creation of ParameterStoreConfiguration
func DefaultParameterStoreConfiguration() (*ParameterStoreConfiguration, error) {
	sess, err := session.NewSession()
	if err != nil {
		return nil, err
	}
	return &ParameterStoreConfiguration{
		SSM: ssm.New(sess),
	}, nil
}
func (p *ParameterStoreConfiguration) getCircuit() Circuit {
	if p.Circuit != nil {
		return p.Circuit
	}
	return defaultCircuit{}
}

func (p *ParameterStoreConfiguration) path(key string) string {
	parts := make([]string, 0, 5)
	pathParts := []string{
		p.Prefix,
		p.Team,
		p.Environment,
		p.Service,
		key,
	}
	for _, part := range pathParts {
		x := strings.Trim(part, "/")
		if x != "" {
			parts = append(parts, x)
		}
	}
	return "/" + strings.Join(parts, "/")
}

// Get should return the given config value.  If the value does not exist, it should return nil, nil.
func (p *ParameterStoreConfiguration) Get(key string) ([]byte, error) {
	path := p.path(key)
	var resp *ssm.GetParametersOutput
	err := p.getCircuit().Run(context.Background(), func(ctx context.Context) error {
		var req *request.Request
		req, resp = p.SSM.GetParametersRequest(&ssm.GetParametersInput{
			WithDecryption: &p.AllowEncrypted,
			Names:          []*string{&path},
		})
		req.SetContext(ctx)
		return req.Send()
	})
	if err != nil {
		return nil, err
	}
	for _, param := range resp.Parameters {
		if !p.AllowEncrypted && *param.Type == "SecureString" {
			continue
		}
		if *param.Name == path {
			distconf.ReaderCacheNotify(&p.ReaderCache, key, []byte(*param.Value))
			return []byte(*param.Value), nil
		}
	}
	distconf.ReaderCacheNotify(&p.ReaderCache, key, nil)
	return nil, nil
}

// Refresh the entire cache, by doing a Get() on all stored keys
func (p *ParameterStoreConfiguration) Refresh() {
	distconf.ReaderCacheRefresh(&p.ReaderCache, p, nil)
}

// Close does nothing
func (p *ParameterStoreConfiguration) Close() {
}
