package main

import (
	"database/sql"
	"errors"
	"os"

	"code.justin.tv/vapour/authproxy/pkg/config"
	"code.justin.tv/vapour/authproxy/pkg/userstore"
	"code.justin.tv/vapour/authproxy/pkg/web"
	gotwitchapi "github.com/dankeroni/gotwitch"
	"github.com/golang-migrate/migrate"
	"golang.org/x/oauth2"
	"golang.org/x/oauth2/twitch"

	_ "github.com/go-sql-driver/mysql"
)

var (
	scopes = []string{}
)

type application struct {
	quitChannel  chan bool
	config       *config.Config
	oauth2Config *oauth2.Config
	api          *gotwitchapi.TwitchAPI
	sqlClient    *sql.DB
	userStore    *userstore.UserStore
	web          *web.Server
}

func newApplication(quitChannel chan bool) *application {
	a := &application{
		quitChannel: quitChannel,
	}

	return a
}

func (a *application) LoadConfig(path string) error {
	var err error

	a.config, err = config.LoadConfig(path)
	if err != nil {
		return err
	}

	// Ensure a password has been set
	if _, ok := os.LookupEnv("VAPOUR_AUTHPROXY_PASSWORD"); !ok {
		return errors.New("Missing authproxy password environment variable: VAPOUR_AUTHPROXY_PASSWORD")
	}

	return nil
}

func (a *application) InitializeOAuth2() error {
	if a.config.App.ClientID == "" {
		return errors.New("Missing required ClientID in config /App/ClientID")
	}

	if a.config.App.ClientSecret == "" {
		return errors.New("Missing required ClientSecret in config /App/ClientSecret")
	}

	if a.config.App.RedirectURL == "" {
		return errors.New("Missing required RedirectURL in config /App/RedirectURL")
	}

	a.oauth2Config = &oauth2.Config{
		ClientID:     a.config.App.ClientID,
		ClientSecret: a.config.App.ClientSecret,
		Scopes:       scopes,
		Endpoint:     twitch.Endpoint,
		RedirectURL:  a.config.App.RedirectURL,
	}

	return nil
}

func (a *application) InitializeAPIClient() error {
	a.api = gotwitchapi.New(a.config.App.ClientID)
	return nil
}

func (a *application) InitializeSQLClient() (err error) {
	if a.config.SQL.DSN == "" {
		return errors.New("Missing required DSN in config /SQL/DSN")
	}

	a.sqlClient, err = sql.Open("mysql", a.config.SQL.DSN)

	return
}

func (a *application) RunMigrations() error {
	m, err := migrate.New("file://./migrations", "mysql://"+a.config.SQL.DSN)
	if err != nil {
		return err
	}

	err = m.Up()
	if err == migrate.ErrNoChange {
		return nil
	}
	return err
}

func (a *application) LoadUsers() error {
	a.userStore = userstore.New(a.sqlClient, a.oauth2Config, a.api)
	return a.userStore.Load()
}

func (a *application) RunWeb() error {
	a.web = web.New(&a.config.Web, a.oauth2Config, a.userStore, versionString)

	go a.web.Run()

	return nil
}
