package web

import (
	"encoding/json"
	"fmt"
	"log"
	"net/http"
	"os"
	"time"

	"code.justin.tv/vapour/authproxy/pkg"
	"code.justin.tv/vapour/authproxy/pkg/config"
	"code.justin.tv/vapour/authproxy/pkg/state"
	"code.justin.tv/vapour/authproxy/pkg/statestore"
	"code.justin.tv/vapour/authproxy/pkg/userstore"
	"github.com/gorilla/mux"
	"golang.org/x/oauth2"

	_ "github.com/golang-migrate/migrate/database/mysql"
	_ "github.com/golang-migrate/migrate/source/file"
)

type Server struct {
	oauth2Config  *oauth2.Config
	userStore     *userstore.UserStore
	versionString string

	r          *mux.Router
	srv        *http.Server
	stateStore *statestore.StateStore

	password string
}

func makeRouter() *mux.Router {
	r := mux.NewRouter()

	pathPrefix := os.Getenv("VAPOUR_AUTHPROXY_PATH_PREFIX")

	if pathPrefix != "" {
		log.Println("Web path prefix:", pathPrefix)
		r = r.PathPrefix(pathPrefix).Subrouter()
	}

	return r
}

func New(cfg *config.WebConfig, oauth2Config *oauth2.Config, userStore *userstore.UserStore, versionString string) *Server {
	password := os.Getenv("VAPOUR_AUTHPROXY_PASSWORD")
	s := &Server{
		oauth2Config:  oauth2Config,
		userStore:     userStore,
		versionString: versionString,
		r:             makeRouter(),
		stateStore:    statestore.New(),

		password: password,
	}

	s.srv = &http.Server{
		Handler:      s.r,
		Addr:         cfg.Host,
		WriteTimeout: 15 * time.Second,
		ReadTimeout:  15 * time.Second,
	}

	s.r.HandleFunc("/version", s.versionHandler)
	s.r.HandleFunc("/auth/login", s.AuthLoginHandler)
	s.r.HandleFunc("/auth/refresh", s.AuthRefreshHandler)
	s.r.HandleFunc("/auth/callback", s.AuthCallbackHandler)

	log.Println("Binding on host", cfg.Host)

	return s
}

func (s *Server) Run() error {
	err := s.srv.ListenAndServe()
	if err != nil {
		fmt.Println("Error in ListenAndServe:", err)
	}
	return nil
}

func (s *Server) versionHandler(w http.ResponseWriter, r *http.Request) {
	w.Write([]byte(s.versionString))
}

func (s *Server) AuthLoginHandler(w http.ResponseWriter, r *http.Request) {
	state, err := state.New()
	if err != nil {
		w.Write([]byte("unable to generate state"))
		return
	}

	s.stateStore.Add(state)

	http.Redirect(w, r, s.oauth2Config.AuthCodeURL(state.Value()), http.StatusTemporaryRedirect)
}

func (s *Server) AuthRefreshHandler(w http.ResponseWriter, r *http.Request) {
	if r.Method != http.MethodPost {
		w.Write([]byte("invalid method, only post is permitted"))
		return
	}

	// TODO: Verify `Authorization` header

	userID := r.FormValue("userID")
	if userID == "" {
		w.Write([]byte("missing userID"))
		return
	}

	password := r.FormValue("password")
	if password != s.password {
		w.Write([]byte("missing userID"))
		return
	}

	if user, ok := s.userStore.Get(userID); ok {
		// Export "public" pieces of a token and send them to the writer

		token, err := user.Token()
		if err != nil {
			fmt.Println("Error getting token:", err)
			return
		}

		pt := &pkg.UserToken{
			UserID:      user.ID(),
			AccessToken: token.AccessToken,
			Expiry:      token.Expiry,
		}

		b, err := json.Marshal(pt)
		if err != nil {
			fmt.Println("Error marshalling usertoken:", err)
			return
		}

		_, err = w.Write(b)
		if err != nil {
			fmt.Println("Error writing response:", err)
			return
		}

		return
	}

	w.Write([]byte("no token found"))
}

func (s *Server) AuthCallbackHandler(w http.ResponseWriter, r *http.Request) {
	if !s.stateStore.Check(r.FormValue("state")) {
		w.Write([]byte("missing/invalid state"))
		return
	}

	err := s.userStore.Add(r.FormValue("code"))
	if err != nil {
		log.Println("Error adding user:", err)
		w.Write([]byte("error adding user"))
		return
	}

	w.Write([]byte("Successfully logged in! /refresh calls for this user will now work"))
}
