#include "browser/browseritem.hpp"
#include "browser/browsermodel.hpp"
#include "debug/log.hpp"

#include <QMenu>
#include <boost/signals2.hpp>

#include <assert.h>

namespace Vape {

BrowserItem::BrowserItem(const QVector<QVariant> &_itemData,
                         BrowserItem *_parent, BrowserModel *_model)
    : identifierPointer(nullptr)
    , identifierId(0)
    , menu(nullptr)
    , parent(_parent)
    , itemData(_itemData)
    , model(_model)
    , editable(false)
{
    if (this->model == nullptr) {
        // If model is null, we try to copy it from the parent
        if (this->parent != nullptr) {
            this->model = this->parent->getModel();
        }
    }

    this->tooltipData.resize(2);
}

BrowserItem::~BrowserItem()
{
    for (auto &child : this->childItems) {
        child->signalBeingRemoved.invoke();
    }

    qDeleteAll(this->childItems);
    this->childItems.clear();
}

BrowserItem *
BrowserItem::getId(int id)
{
    auto match = this->idChildren[id];

    return match;
}

BrowserItem *
BrowserItem::getPath(const char *path)
{
    auto match = this->pathChildren[path];

    return match;
}

BrowserItem *
BrowserItem::createBasic(const QString &key, const QVariant &value)
{
    return this->addBasic(key, value);
}

BrowserItem *
BrowserItem::createPath(const char *path, const QString &key,
                        const QVariant &value)
{
    // Make sure the path does not already exist
    assert(this->pathChildren[path] == nullptr);

    return this->addPath(path, key, value);
}

BrowserItem *
BrowserItem::createPointer(const void *pointer, const QString &key,
                           const QVariant &value)
{
    {
        // Make sure the identifier doesn't already exist
        assert(this->find(pointer, 0) == nullptr);
    }

    return this->addPointer(key, value, pointer);
}

BrowserItem *
BrowserItem::createId(int id, const QString &key, const QVariant &value)
{
    {
        // Make sure the id isn't already in use
        assert(this->idChildren[id] == nullptr);
    }

    return this->addId(key, value, id);
}

BrowserItem *
BrowserItem::getPointer(const void *pointer)
{
    BrowserItem *match = this->find(pointer, 0);

    return match;
}

void
BrowserItem::setExpanded(bool expanded)
{
    this->model->setExpanded(this, expanded);
}

BrowserItem *
BrowserItem::set(const char *path, const QString &key, const QString &value)
{
    auto item = this->pathChildren[path];

    if (item == nullptr) {
        this->createPath(path, key, value);
    }

    return this;
}

BrowserItem *
BrowserItem::increment(const char *path, int amount)
{
    BrowserItem *item = this->getPath(path);

    item->setData(1, item->getData(1).toInt() + amount);

    return this;
}

BrowserItem *
BrowserItem::setValue(const char *path, const QVariant &value)
{
    auto item = this->getPath(path);

    if (item != nullptr) {
        item->setValue(value);
    }

    return this;
}

BrowserItem *
BrowserItem::setValueById(int id, const QVariant &value)
{
    auto item = this->getId(id);

    item->setValue(value);

    return this;
}

BrowserItem *
BrowserItem::setValue(const QVariant &value, int column)
{
    // Sanity check
    assert(column >= 0);
    assert(column <= 1);

    this->setData(column, value);

    // Signal that the value was changed outside of the GUI
    if (column == 1) {
        std::vector<boost::signals2::shared_connection_block> blocks;
        for (auto connection : this->signalConnections) {
            blocks.emplace_back(connection);
        }
        this->signalValueChanged(value, false);
    }

    return this;
}

BrowserItem *
BrowserItem::addBasic(const QString &key, const QVariant &value)
{
    auto item = this->add(key, value);

    this->unmanagedChildren.push_back(item);

    return item;
}

BrowserItem *
BrowserItem::addPath(const std::string &path, const QString &key,
                     const QVariant &value)
{
    auto item = this->add(key, value);

    item->identifierPath = path;

    this->pathChildren[path] = item;

    return item;
}

BrowserItem *
BrowserItem::addPointer(const QString &key, const QVariant &value,
                        const void *pointer)
{
    auto item = this->add(key, value);

    item->identifierPointer = pointer;

    this->unmanagedChildren.push_back(item);

    return item;
}

BrowserItem *
BrowserItem::addId(const QString &key, const QVariant &value, int id)
{
    auto item = this->add(key, value);

    item->identifierId = id;

    this->idChildren[id] = item;

    return item;
}

BrowserItem *
BrowserItem::add(const QVariant &key, const QVariant &value)
{
    QVector<QVariant> items;
    items << key << value;

    auto item = new BrowserItem(items, this, this->model);

    this->appendChild(item);

    return item;
}

bool
BrowserItem::removePointer(const void *pointer)
{
    BrowserItem *match = this->find(pointer, 0);
    if (match == nullptr) {
        return false;
    }

    this->removeChild(match);

    // Remove from children or unmanagedChildren
    if (match->identifierPath.length() > 0) {
        this->pathChildren[match->identifierPath] = nullptr;
    } else {
        this->unmanagedChildren.erase(
            std::remove_if(std::begin(this->unmanagedChildren),
                           std::end(this->unmanagedChildren),
                           [match](BrowserItem *x) { return (x == match); }),
            std::end(this->unmanagedChildren));
    }

    return true;
}

bool
BrowserItem::removeId(int id)
{
    BrowserItem *match = this->idChildren[id];
    if (match == nullptr) {
        return false;
    }

    this->removeChild(match);

    this->idChildren[id] = nullptr;

    return true;
}

bool
BrowserItem::removeBrowserItem(BrowserItem *browserItem)
{
    if (!this->removeChild(browserItem)) {
        return false;
    }

    for (auto it = std::begin(this->pathChildren);
         it != std::end(this->pathChildren);) {
        if (it->second == browserItem) {
            this->pathChildren.erase(it);
            return true;
        } else {
            ++it;
        }
    }

    for (auto it = std::begin(this->idChildren);
         it != std::end(this->idChildren);) {
        if (it->second == browserItem) {
            this->idChildren.erase(it);
            return true;
        } else {
            ++it;
        }
    }

    this->unmanagedChildren.erase(std::find(std::begin(this->unmanagedChildren),
                                            std::end(this->unmanagedChildren),
                                            browserItem));

    return true;
}

void
BrowserItem::clear()
{
    if (this->model) {
        emit this->model->layoutAboutToBeChanged();
    }

    for (auto &child : this->childItems) {
        child->signalBeingRemoved.invoke();
    }

    qDeleteAll(this->childItems);
    this->childItems.clear();

    if (this->model) {
        emit this->model->layoutChanged();
    }

    this->pathChildren.clear();
    this->unmanagedChildren.clear();
}

void
BrowserItem::appendChild(BrowserItem *item)
{
    if (this->model) {
        emit this->model->layoutAboutToBeChanged();
    }

    this->childItems.append(item);

    if (this->model) {
        emit this->model->layoutChanged();
    }
}

bool
BrowserItem::removeChild(BrowserItem *item)
{
    if (this->model) {
        emit this->model->layoutAboutToBeChanged();
    }

    Log::debugf("Remove child");
    item->signalBeingRemoved.invoke();

    bool ret = this->childItems.removeOne(item);

    if (this->model) {
        emit this->model->layoutChanged();
    }

    if (ret) {
        // ??????
        // delete item;
    }

    return ret;
}

BrowserItem *
BrowserItem::child(int row)
{
    // Sanity check
    assert(row >= 0);

    return this->childItems.value(row);
}

int
BrowserItem::childCount() const
{
    return this->childItems.count();
}

int
BrowserItem::columnCount() const
{
    return this->itemData.count();
}

QVariant
BrowserItem::getData(int column, int role) const
{
    if (role == Qt::DisplayRole) {
        switch (this->userData.type) {
            case UserData::Type::ComboBox: {
                if (column == 1) {
                    auto index = this->userData.itemModel->index(
                        this->itemData.value(column).value<int>(), 0);
                    return this->userData.itemModel->data(index);
                } else {
                    return this->itemData.value(column);
                }
            } break;
        }
    } else if (role == Qt::ToolTipRole) {
        return this->tooltipData.value(column);
    }

    return this->itemData.value(column);
}

int
BrowserItem::getRow() const
{
    if (this->parent) {
        return this->parent->childItems.indexOf(
            const_cast<BrowserItem *>(this));
    }

    return 0;
}

BrowserItem *
BrowserItem::getParent()
{
    return this->parent;
}

bool
BrowserItem::setData(int column, const QVariant &value, int role)
{
    if (column < 0) {
        return false;
    }

    if (role == Qt::ToolTipRole) {
        if (column > this->tooltipData.size()) {
            return false;
        }

        this->tooltipData[column] = value;

        return true;
    }

    if (column > this->itemData.size()) {
        return false;
    }

    this->itemData[column] = value;

    if (this->model) {
        auto index =
            this->model->getIndexByPointer(this->getRow(), column, this);
        emit this->model->dataChanged(index, index);
    }

    return true;
}

bool
BrowserItem::setDataFromEditor(int column, const QVariant &value)
{
    if (!this->setData(column, value)) {
        return false;
    }

    this->signalValueChanged(value, true);

    return true;
}

BrowserItem *
BrowserItem::find(const void *needle, int depth)
{
    for (BrowserItem *innerItem : this->unmanagedChildren) {
        if (innerItem->identifierPointer == needle) {
            return innerItem;
        }

        if (depth > 0 && innerItem->childCount() > 0) {
            BrowserItem *ret = innerItem->find(needle, depth - 1);
            if (ret != nullptr) {
                return ret;
            }
        }
    }

    return nullptr;
}

BrowserModel *
BrowserItem::getModel() const
{
    return this->model;
}

void
BrowserItem::setEditable(bool newEditable)
{
    this->editable = newEditable;

    // XXX(pajlada): Do we need to emit some sort of QT signal here saying that
    // a flag has changed?
}

bool
BrowserItem::isEditable() const
{
    return this->editable;
}

void
Receiver::onActionTriggered(QAction *action)
{
    auto it = this->functions.find(action);
    if (it == std::end(this->functions)) {
        return;
    }

    BIAction f = this->functions[action];
    f.first(f.second);
}

static Receiver receiver;

BrowserItem *
BrowserItem::addAction(const QString &text, const BIFunction &function)
{
    if (this->menu == nullptr) {
        this->menu = new QMenu(nullptr);

        QObject::connect(this->menu, &QMenu::triggered, &receiver,
                         &Receiver::onActionTriggered);
    }

    auto action = this->menu->addAction(text);
    receiver.functions[action] = std::make_pair(function, this);

    return this;
}

}  // namespace Vape
