#pragma once

#include "fundamentals/observable.hpp"

#include <boost/signals2/signal.hpp>
#include <pajlada/signals/signal.hpp>

#include <deque>
#include <unordered_map>

class QMenu;

namespace Vape {

class BrowserItem;

typedef std::function<void(BrowserItem *)> BIFunction;
typedef std::pair<BIFunction, BrowserItem *> BIAction;

class Receiver : public QObject
{
public:
    std::unordered_map<QAction *, BIAction> functions;

public slots:
    void onActionTriggered(QAction *action);
};

class BrowserModel;

class BrowserItem
{
public:
    explicit BrowserItem(const QVector<QVariant> &_itemData,
                         BrowserItem *_parent, BrowserModel *_model = nullptr);
    ~BrowserItem();

    // Get an object a certain path
    // Returns object at path or nullptr if it doesn't exist
    BrowserItem *getId(int identifierId);

    // Get an object at a certain path
    // Returns object at path, or nullptr if it doesn't exist
    BrowserItem *getPath(const char *identifierPath);

    // Create an object without any identfying object
    // Returns created object
    BrowserItem *createBasic(const QString &key,
                             const QVariant &value = QVariant());

    // Create an object with a certain path
    // Returns created object
    // Asserts if path is in use
    BrowserItem *createPath(const char *identifierPath, const QString &key,
                            const QVariant &value = QVariant());

    // Create an object with a certain pointer
    // Returns created object
    BrowserItem *createPointer(const void *identifierPointer,
                               const QString &key,
                               const QVariant &value = QVariant());

    // Create an object with a certain id
    // Returns created object
    BrowserItem *createId(int identifierId, const QString &key,
                          const QVariant &value = QVariant());

    // Get or create an object with a certain pointer
    // Returns object with needle or nullptr if it doesn't exist
    BrowserItem *getPointer(const void *identifierPointer);

    // Expands/collapses this item
    void setExpanded(bool expanded);

    // Set the text of an object with a certain path
    // This will create the object if it does not already exist
    // Returns this
    BrowserItem *set(const char *identifierPath, const QString &key,
                     const QString &value);

    // Increment value at path by `amount`
    // Returns this
    BrowserItem *increment(const char *identifierPath, int amount);

    // Set the text of an object with a certain path
    // This will create the object if it does not already exist
    // Returns this
    BrowserItem *setValue(const char *identifierPath, const QVariant &value);

    // Set the text of an object with a certain path
    // This will create the object if it does not already exist
    // Returns this
    BrowserItem *setValueById(int identifierId, const QVariant &value);

    // Set value of this (value = text displayed aka right column)
    // Returns this
    BrowserItem *setValue(const QVariant &value, int column = 1);

    // Remove an object with a certain pointer
    bool removePointer(const void *identifierPointer);

    // Remove an object with a certain id
    bool removeId(int identifierId);

    // Remove an object by the direct BrowserItem pointer
    bool removeBrowserItem(BrowserItem *browserItem);

    // TODO(pajlada): implement removePath if we need it

    // Clear all child objects
    void clear();

    // Returns the child item at the given row/index
    BrowserItem *child(int getRow);

    template <typename T>
    void
    registerToObservable(Observable<T> &observable)
    {
        this->signalConnections.push_back(
            observable.signalValueChanged->connect([this](const T &newValue) {
                this->setValue(QVariant::fromValue(newValue));
            }));

        this->setEditable(true);

        this->signalValueChanged.connect(
            [&observable](QVariant newValue, bool /*fromGUI*/) mutable {
                observable = newValue.value<T>();
            });
    }

    std::deque<boost::signals2::connection> signalConnections;

    // Returns the amount of children of this item
    int childCount() const;

    // Returns the amount of column of this item
    int columnCount() const;

    // Returns the row at which this item is located if it has a parent
    int getRow() const;

    // Returns parent item
    // Can be nullptr
    BrowserItem *getParent();

    // Return data at given column
    QVariant getData(int column, int role = Qt::DisplayRole) const;

    // Set column data
    // Returns false if the column was out of bounds
    // Otherwise, sets data and returns true
    bool setData(int column, const QVariant &value, int role = Qt::DisplayRole);

    bool setDataFromEditor(int column, const QVariant &value);

    // Find a browser item by an identifier needle by children
    BrowserItem *find(const void *needle, int depth = 10);

    // Returns stored model
    BrowserModel *getModel() const;

    // Getter/Setter for the "editable" flag
    void setEditable(bool newEditable);
    bool isEditable() const;

    /// Menu code
    // addAction returns this for chaining
    BrowserItem *addAction(const QString &text, const BIFunction &function);

private:
    // Various identificiation methods
    const void *identifierPointer;
    uint32_t identifierId;
    std::string identifierPath;

public:
    uint32_t
    getIdentifierId() const
    {
        return this->identifierId;
    }

    const std::string &
    getIdentifierPath() const
    {
        return this->identifierPath;
    }

    /// Signals
    // This signal is invoked whenever the BrowserItem value(1) is changed
    boost::signals2::signal<void(QVariant, bool)> signalValueChanged;

    // This signal is invoked whenever the signal is about to be removed
    // It will be invoked from the parent BrowserItem
    pajlada::Signals::NoArgBoltSignal signalBeingRemoved;

    QMenu *menu;

    struct UserData {
        enum class Type {
            None,
            ComboBox,
        } type = Type::None;

        union {
            QAbstractItemModel *itemModel;
        };
    } userData;

private:
    // Add child object without any identifying object
    // Return created object
    BrowserItem *addBasic(const QString &key, const QVariant &value);

    // Add child object with path
    // Return created object
    BrowserItem *addPath(const std::string &identifierPath, const QString &key,
                         const QVariant &value);

    // Add child object with pointer
    // Return created object
    BrowserItem *addPointer(const QString &key, const QVariant &value,
                            const void *identifierPointer);

    // Add child object with id
    // Return created object
    BrowserItem *addId(const QString &key, const QVariant &value,
                       int identifierId);

    // Create and append an item with a certain text
    // Returns created object
    BrowserItem *add(const QVariant &key, const QVariant &value);

    // Append given child as a new row
    void appendChild(BrowserItem *child);

    // Remove given child
    bool removeChild(BrowserItem *child);

    // Parent row
    BrowserItem *parent;

    // List of children (rows)
    QList<BrowserItem *> childItems;

    // Vector of columns
    QVector<QVariant> itemData;

    QVector<QVariant> tooltipData;

    // Pointer to base model
    BrowserModel *model;

    // Decides whether the value column of this row is editable
    bool editable;

    // Children that are accessible with a path
    std::unordered_map<std::string, BrowserItem *> pathChildren;

    // Children that are accessible with an id
    std::unordered_map<uint32_t, BrowserItem *> idChildren;

    // Children that are unaccessible with a path. Currently stores children
    // who are identified by a pointer not at all
    std::deque<BrowserItem *> unmanagedChildren;
};

}  // namespace Vape
