#include "browser/browsermodel.hpp"
#include "browser/browseritem.hpp"
#include "network/transform.hpp"

#include <QHeaderView>
#include <QMenu>
#include <QTreeView>

namespace Vape {

BrowserModel::BrowserModel()
    : QAbstractItemModel(nullptr)
{
    QVector<QVariant> headers;
    headers << "Key"
            << "Value";

    this->rootItem = new BrowserItem(headers, nullptr, this);
}

BrowserModel::~BrowserModel()
{
}

void
BrowserModel::setTreeView(QTreeView *_treeView)
{
    this->treeView = _treeView;

    this->treeView->setModel(this);
    this->treeView->setContextMenuPolicy(Qt::CustomContextMenu);

    connect(this->treeView, &QTreeView::customContextMenuRequested, this,
            &BrowserModel::onContextMenu);

    this->treeView->header()->setSectionResizeMode(0, QHeaderView::Stretch);
    this->treeView->header()->setSectionResizeMode(1, QHeaderView::Stretch);
}

void
BrowserModel::createRoot(const char *path, const QString &key,
                         const QVariant &value)
{
    this->rootItem->createPath(path, key, value);
}

BrowserItem *
BrowserModel::getRoot(const char *path)
{
    return this->rootItem->getPath(path);
}

QModelIndex
BrowserModel::getIndexByPointer(int row, int column, void *ptr)
{
    return this->createIndex(row, column, ptr);
}

void
BrowserModel::setExpanded(BrowserItem *browserItem, bool expanded)
{
    if (this->treeView) {
        auto index =
            this->getIndexByPointer(browserItem->getRow(), 0, browserItem);
        if (index.isValid()) {
            this->treeView->setExpanded(index, expanded);
        }
    }
}

Qt::ItemFlags
BrowserModel::flags(const QModelIndex &index) const
{
    if (!index.isValid()) {
        return 0;
    }

    if (index.column() == 1) {
        auto browserItem = this->getItem(index);

        if (browserItem->isEditable()) {
            return Qt::ItemIsEditable | QAbstractItemModel::flags(index);
        }
    }

    return QAbstractItemModel::flags(index);
}

QVariant
BrowserModel::data(const QModelIndex &index, int role) const
{
    if (!index.isValid()) {
        return QVariant();
    }

    if (role != Qt::DisplayRole && role != Qt::EditRole &&
        role != Qt::ToolTipRole) {
        return QVariant();
    }

    auto browserItemRow = this->getItem(index);
    auto data = browserItemRow->getData(index.column(), role);

    return data;
}

bool
BrowserModel::setData(const QModelIndex &parent, const QVariant &value,
                      int role)
{
    if (role != Qt::EditRole) {
        return false;
    }

    BrowserItem *item = this->getItem(parent);

    int col = parent.column();

    if (item->setData(col, value)) {
        if (col == 1) {
            // Signal that the value was changed from the GUI
            item->signalValueChanged(value, true);
        }

        emit this->dataChanged(parent, parent);

        if (this->treeView != nullptr) {
            this->treeView->resizeColumnToContents(0);
            this->treeView->resizeColumnToContents(1);
        }

        return true;
    }

    return false;
}

QVariant
BrowserModel::headerData(int section, Qt::Orientation orientation,
                         int role) const
{
    if (orientation == Qt::Horizontal && role == Qt::DisplayRole) {
        return rootItem->getData(section);
    }

    return QVariant();
}

bool
BrowserModel::hasChildren(const QModelIndex &parent) const
{
    BrowserItem *parentItem = this->getItem(parent);

    int numChildren = parentItem->childCount();

    return (numChildren > 0);
}

QModelIndex
BrowserModel::index(int row, int column, const QModelIndex &parent) const
{
    if (parent.isValid() && parent.column() != 0) {
        return QModelIndex();
    }

    BrowserItem *parentItem = this->getItem(parent);

    BrowserItem *childItem = parentItem->child(row);
    if (childItem) {
        return this->createIndex(row, column, childItem);
    }

    return QModelIndex();
}

QModelIndex
BrowserModel::parent(const QModelIndex &index) const
{
    if (!index.isValid()) {
        return QModelIndex();
    }

    auto childItem = this->getItem(index);
    auto parentItem = childItem->getParent();

    if (parentItem == rootItem) {
        return QModelIndex();
    }

    return this->createIndex(parentItem->getRow(), 0, parentItem);
}

int
BrowserModel::rowCount(const QModelIndex &parent) const
{
    return this->getItem(parent)->childCount();
}

int
BrowserModel::columnCount(const QModelIndex &parent) const
{
    if (parent.isValid()) {
        auto browserItem = static_cast<BrowserItem *>(parent.internalPointer());
        return browserItem->columnCount();
    }

    return rootItem->columnCount();
}

void
BrowserModel::onContextMenu(const QPoint &point)
{
    QModelIndex index = this->treeView->indexAt(point);
    if (index.isValid()) {
        auto browserItem = static_cast<BrowserItem *>(index.internalPointer());
        if (browserItem->menu != nullptr) {
            browserItem->menu->exec(this->treeView->mapToGlobal(point));
        }
    }
}

BrowserItem *
BrowserModel::getItem(const QModelIndex &index) const
{
    if (index.isValid()) {
        BrowserItem *item = static_cast<BrowserItem *>(index.internalPointer());

        if (item) {
            return item;
        }
    }

    return rootItem;
}

}  // namespace Vape
