#pragma once

#include <QAbstractItemModel>
#include <QModelIndex>
#include <QVariant>

class QTreeView;

namespace Vape {

class BrowserItem;

class BrowserModel : public QAbstractItemModel
{
    Q_OBJECT

public:
    BrowserModel();
    ~BrowserModel();

    void setTreeView(QTreeView *_treeView);

    /** Methods that should be accessible from outside **/
    // Add a root object with the given path and key/value pair
    void createRoot(const char *path, const QString &key,
                    const QVariant &value);

    // Get root object at given path
    BrowserItem *getRoot(const char *path);

    /** Custom methods that are accessible from within Browser **/
    // Return a QModelIndex from a row, column and internal pointer
    // It just calls and returns the value from model->createIndex
    QModelIndex getIndexByPointer(int row, int column, void *ptr);

    // Sets the expanded value of the given browser item
    void setExpanded(BrowserItem *browserItem, bool expanded);

    /** QAbstractItemModel overloads **/
    // Return flags for object at given index
    // flags can decide if the index is editable or not, for example
    Qt::ItemFlags flags(const QModelIndex &index) const Q_DECL_OVERRIDE;

    // data is used to display things
    QVariant data(const QModelIndex &index, int role) const Q_DECL_OVERRIDE;

    // setData is called when an edit has been made
    bool setData(const QModelIndex &parent, const QVariant &value,
                 int role = Qt::EditRole) Q_DECL_OVERRIDE;

    // Return data for header
    // NOTE: We've only implemented horizontal header support, we shouldn't have
    // to implement vertical header support
    QVariant headerData(int section, Qt::Orientation orientation,
                        int role = Qt::DisplayRole) const Q_DECL_OVERRIDE;

    // Returns true if object at given index `parent` has any children
    bool hasChildren(const QModelIndex &parent) const Q_DECL_OVERRIDE;

    // Return index of object at given row/column with given parent
    QModelIndex index(int row, int column,
                      const QModelIndex &parent = QModelIndex()) const
        Q_DECL_OVERRIDE;

    // Return parent of object at the given index
    QModelIndex parent(const QModelIndex &index) const Q_DECL_OVERRIDE;

    // Return number of rows/children under the given parent object
    int rowCount(const QModelIndex &parent = QModelIndex()) const
        Q_DECL_OVERRIDE;

    // Return number of columns for the children of the given parent object
    int columnCount(const QModelIndex &parent = QModelIndex()) const
        Q_DECL_OVERRIDE;

public slots:
    void onContextMenu(const QPoint &point);

private:
    // Returns object at given index
    BrowserItem *getItem(const QModelIndex &index) const;

    BrowserItem *rootItem;

    QTreeView *treeView;
};

}  // namespace Vape
