#!/bin/sh

set -e

usage() {
    >&2 echo "Usage: $0 [--library|--application] target_dir"
    >&2 echo " * Example for an application: $0 --application /path/to/project"
    >&2 echo " * Example for a library: $0 --library /path/to/project"
    exit 1
}

# Install any scripts required for a C++ application
source_app_name="cpp-skeleton"
project_subtype=""

source_utils_dir=$(CDPATH='' cd -- "$(dirname -- "$0")" && pwd)
source_docker_dir="$source_utils_dir/../.docker"

. "$source_utils_dir/"ci-test-utils/shared.sh

for i in "$@"; do
	case $i in
		--library)
			project_subtype="library"
			shift
			;;
		--application)
			project_subtype="application"
			shift
			;;
		--help)
            usage
			shift
			;;
		*)
			# unknown option
			;;
	esac
done

[ "$project_subtype" != "application" ] && [ "$project_subtype" != "library" ] && usage

source_dir="$(realpath "$source_utils_dir/..")"
target_dir=$1

verify_target_dir "$target_dir" || usage

target_app_name=$(basename "$(emulate_realpath "$target_dir")")
target_utils_dir="$target_dir/utils"
target_utils_docker_dir="$target_dir/utils/docker"
target_docker_dir="$target_dir/.docker"
target_docs_dir="$target_dir/docs"

echo "Installing utils to target directory '$target_dir' ($target_app_name)"

create_directory "$target_utils_dir"
create_directory "$target_utils_docker_dir"
create_directory "$target_docker_dir"
create_directory "$target_docs_dir"

if ask_override_file "$source_dir/.gitignore" "$target_dir"; then
    safe_cp "$source_path" "$target_path"
fi

if ask_override_file "$source_dir/.clang-format" "$target_dir"; then
    safe_cp "$source_path" "$target_path"
fi

if ask_override_file "$source_dir/.clang-tidy" "$target_dir"; then
    safe_cp "$source_path" "$target_path"
fi

if [ "$project_subtype" = "application" ]; then
    safe_cp "$source_utils_dir/build.sh" "$target_utils_dir/"
    safe_cp "$source_utils_dir/deploy.sh" "$target_utils_dir/"
    safe_cp "$source_utils_dir/deploy-symbols.sh" "$target_utils_dir/"

    if ask_override_file "$source_utils_dir/pre-deploy-symbols.sh" "$target_utils_dir"; then
        cp "$source_path" "$target_path"
    fi

    if ask_override_file "$source_dir/jenkins.groovy" "$target_dir"; then
        tmp_file=$(mktemp)
        sed "s/$source_app_name/$target_app_name/g" "$source_path" > "$tmp_file"
        mv "$tmp_file" "$target_path"
    fi

    if ask_override_file "$source_dir/.manta.json" "$target_dir"; then
        tmp_file=$(mktemp)
        sed "s/$source_app_name/$target_app_name/g" "$source_path" > "$tmp_file"
        mv "$tmp_file" "$target_path"
    fi
fi

if [ "$project_subtype" = "library" ]; then
    safe_cp "$source_utils_dir/example.sh" "$target_utils_dir/build.sh"

    tmp_file=$(mktemp)
    source_path="$source_dir/files/library-jenkins.groovy"
    target_path="$target_dir/jenkins.groovy"
    sed "s/$source_app_name/$target_app_name/g" "$source_path" > "$tmp_file"
    mv "$tmp_file" "$target_path"

    tmp_file=$(mktemp)
    source_path="$source_dir/files/library-manta.json"
    target_path="$target_dir/.manta.json"
    sed "s/$source_app_name/$target_app_name/g" "$source_path" > "$tmp_file"
    mv "$tmp_file" "$target_path"
fi

if ask_override_file "$source_utils_dir/config.sh" "$target_utils_dir"; then
    tmp_file=$(mktemp)
    sed "s/$source_app_name/$target_app_name/g" "$source_path" > "$tmp_file"
    mv "$tmp_file" "$target_path"
fi

safe_cp "$source_utils_dir/setup.sh" "$target_utils_dir/"
safe_cp "$source_utils_dir/test.sh" "$target_utils_dir/"
safe_cp "$source_utils_dir/manta.sh" "$target_utils_dir/"
safe_cp "$source_utils_dir/update-internal.sh" "$target_utils_dir/"
safe_cp "$source_utils_dir/safe-exec.sh" "$target_utils_dir/"

safe_cp "$source_utils_dir/docker/image.sh" "$target_utils_docker_dir/"
safe_cp "$source_utils_dir/docker/build.sh" "$target_utils_docker_dir/"
safe_cp "$source_utils_dir/docker/test.sh" "$target_utils_docker_dir/"
safe_cp "$source_utils_dir/docker/push.sh" "$target_utils_docker_dir/"
safe_cp "$source_utils_dir/docker/run.sh" "$target_utils_docker_dir/"

[ ! -f "$target_docker_dir/Dockerfile.build" ] && safe_cp "$source_docker_dir/Dockerfile.build" "$target_docker_dir/Dockerfile.build"
[ ! -f "$target_docker_dir/Dockerfile.run" ] && safe_cp "$source_docker_dir/Dockerfile.run" "$target_docker_dir/Dockerfile.run"
[ ! -f "$target_dir/.licenses.json" ] && safe_cp "$source_dir/.licenses.json" "$target_dir/.licenses.json"

# Documentation
safe_cp "$source_dir/docs/VERSIONING.md" "$target_docs_dir/"
safe_cp "$source_dir/docs/CI.md" "$target_docs_dir/"
safe_cpr "$source_dir/docs/graphs/" "$target_docs_dir/"

# GitHub helper things
safe_cpr "$source_dir/.github" "$target_dir/"

# Bootstrap googletest
if [ ! -d "$target_dir/external/googletest" ]; then
    echo " * Bootstrapping googletest"
    mkdir -p "$target_dir/external"
    ( cd "$target_dir/external" && git submodule add --force https://github.com/google/googletest )
fi

# Ensure googletest is on the right version
if [ -d "$target_dir/external/googletest" ]; then
    googletest_hash=$(git -C "$target_dir/external/googletest" rev-parse HEAD)

    googletest_target_hash="e2239ee6043f73722e7aa812a459f54a28552929"
    googletest_target_tag="release-1.11.0"

    if [ "$googletest_hash" != "$googletest_target_hash" ]; then
        echo " * Upgrading googletest to $googletest_target_tag"
        ( cd "$target_dir/external/googletest" && git fetch && git checkout "$googletest_target_tag" 2>/dev/null )
    fi
fi

# Bootstrap tests
if [ ! -d "$target_dir/tests" ]; then
    echo " * Bootstrapping tests"
    mkdir "$target_dir/tests"
    mkdir "$target_dir/tests/src"

    tmp_file=$(mktemp)
    sed "s/$source_app_name/$target_app_name/g" "$source_dir/tests/CMakeLists.txt" > "$tmp_file"
    mv "$tmp_file" "$target_dir/tests/CMakeLists.txt"

    safe_cp "$source_dir/tests/src/main.cpp" "$target_dir/tests/src/main.cpp"
fi
