package main

import (
	"flag"
	"fmt"
	"strings"

	"github.com/aws/aws-sdk-go/aws/awserr"
	"github.com/aws/aws-sdk-go/service/s3"

	"code.justin.tv/vapour/deployer/vapour"
)

const (
	bucketName = "twitch-vapour-builds"

	currentVersionLocalPath = "currentVersion.txt"
	projectFilesLocalPath   = "projectFiles.txt"
)

// Valid targets to deploy to
var validTargets = []string{
	"bridge",
	"tracker",
	"plugin",
	"test",
}

// The value that will ultimately be checked if we need to run a diff on all "changed files"
var doDiffFiles = true

/// Flags
// --target
var target string

// --changed a --changed b
var changedFiles vapour.ArrayFlags

// --nodiff
var noDiff bool

func init() {
	// Initialize flags
	flag.StringVar(&target, "target", "", "Target (i.e. bridge, tracker, plugin, test)")
	flag.Var(&changedFiles, "changed", "Changed files (mark at least one)")
	flag.BoolVar(&noDiff, "nodiff", false, "Always push specified files, don't do a diff check")
}

func main() {
	// Parse flags & work out final values
	flag.Parse()
	if noDiff {
		doDiffFiles = false
	}

	// Ensure that the given target is valid
	if !vapour.StringInSlice(target, validTargets) {
		fmt.Println("Invalid target")
		return
	}

	// If --changed has not been used at all, we will load files from projectFiles.txt
	if len(changedFiles) == 0 {
		err := changedFiles.AddFromPath(projectFilesLocalPath)
		if err != nil {
			fmt.Println("Error when trying to add files from projectFiles.txt", err)
			return
		}

		// Remove any duplicates (incase we want to allow this together with --changed later)
		changedFiles.RemoveDuplicates()
	}

	// If no files end up being marked as changed, we can't do anything
	if len(changedFiles) == 0 {
		fmt.Println("You must mark at least one file as changed")
		fmt.Println("Usage: ./deployer --changed tracker.exe --changed secondfile.txt OR fill in projectFiles.txt")
		return
	}

	// Create S3 service
	svc, err := vapour.CreateService(bucketName, target)
	if err != nil {
		fmt.Println("Error initializing S3:", err)
		return
	}

	if doDiffFiles {
		// Check diffs
		tmpPaths := []string{}
		actuallyChangedFiles := []string{}

		defer func() {
			// Remove all temporary files after we're done
			vapour.RemoveFiles(tmpPaths)
		}()

		for _, file := range changedFiles {
			localPath := file
			s3Path := svc.GenerateS3Path(localPath)
			remotePath, err := svc.DownloadTemporary(s3Path)
			equal := true

			if err != nil {
				if aerr, ok := err.(awserr.Error); ok {
					if aerr.Code() == s3.ErrCodeNoSuchKey {
						equal = false
					} else {
						fmt.Println("Error downloading temporary file", err)
						return
					}
				} else {
					fmt.Println("Error downloading temporary file", err)
					return
				}
			}

			tmpPaths = append(tmpPaths, remotePath)

			if equal {
				equal, err = vapour.DeepCompare(localPath, remotePath)
				if err != nil {
					fmt.Println("Unable to compare files", err)
					continue
				}
			}

			if !equal {
				actuallyChangedFiles = append(actuallyChangedFiles, localPath)
			}
		}

		changedFiles = actuallyChangedFiles

		if len(changedFiles) == 0 {
			fmt.Println("No files have been changed.")
			fmt.Println("If you added a new file, remember to update projectFiles.txt")
			return
		}
	}

	fmt.Println("The following files will be in the update:")
	for _, localPath := range changedFiles {
		fmt.Println(localPath)
	}

	for {
		fmt.Printf("Do you want to try to push this update? (y/n) ")
		input, err := vapour.GetInputString()
		if err != nil {
			fmt.Println("Error reading user input", err)
			return
		}
		input = strings.ToLower(strings.TrimSpace(input))
		if input == "y" {
			break
		} else if input == "n" {
			return
		}
	}

	changes, err := svc.GenerateChanges(changedFiles)
	if err != nil {
		fmt.Println("Error generating changes:", err)
		return
	}

	currentVersion := vapour.GetCurrentVersion(currentVersionLocalPath)
	latestVersion := svc.GetLatestVersion()

	if currentVersion < latestVersion {
		fmt.Println("You are behind latest version. Make sure you're up to date already!")
		return
	} else if currentVersion > latestVersion {
		fmt.Println("You are ahead of latest version. Something is wrong. Fix it pajlada BabyRage")
		return
	}

	// Increase version by 1 FeelsGoodMan
	newVersion := latestVersion + 1
	fmt.Println("Beginning to deploy", strings.Title(target), "version", newVersion)

	err = svc.UploadChanges(changes)
	if err != nil {
		fmt.Println("Error uploading changes:", err)
		return
	}

	err = svc.UploadVersionFile(changes, newVersion)
	if err != nil {
		fmt.Println("Error uploading changes:", err)
		return
	}

	// Update version locally
	err = vapour.UpdateCurrentVersion(currentVersionLocalPath, newVersion)
	if err != nil {
		fmt.Println("Error updating current version:", err)
		return
	}

	// Update version in s3 and locally
	err = svc.UpdateLatestVersion(newVersion)
	if err != nil {
		fmt.Println("Error updating latest version:", err)
		return
	}

	fmt.Println("Successfully deployed version", newVersion)
}
