package vapour

import (
	"bufio"
	"bytes"
	"fmt"
	"os"
	"strings"
)

// StringInSlice returns true if the given string `a` is in the given string slice `list`
func StringInSlice(a string, list []string) bool {
	for _, b := range list {
		if b == a {
			return true
		}
	}
	return false
}

// ArrayFlags is a helper type for a string slice so it can be used as a flag
type ArrayFlags []string

// String returns a string representation of the array flags
// This is used in the flag interface
func (i *ArrayFlags) String() string {
	return "xd"
}

// Set appends a string value to itself
// This is used in the flag interface
func (i *ArrayFlags) Set(value string) error {
	*i = append(*i, value)
	return nil
}

// AddFromPath adds a list of strings from a given file (newline separated)
func (i *ArrayFlags) AddFromPath(path string) error {
	file, err := os.Open(path)
	if err != nil {
		// Unable to open projectFiles.txt. This just means we rely on --changed instead
		return err
	}

	scanner := bufio.NewScanner(file)
	for scanner.Scan() {
		t := strings.TrimSpace(scanner.Text())
		if len(t) == 0 {
			// Empty line, skip
			continue
		}
		if t[0] == '#' {
			// Comment, skip
			continue
		}
		*i = append(*i, t)
	}

	return file.Close()
}

// RemoveDuplicates remove any duplicate strings from the slice
func (i *ArrayFlags) RemoveDuplicates() {
	seen := make(map[string]struct{}, len(*i))
	j := 0
	for _, v := range *i {
		if _, ok := seen[v]; ok {
			continue
		}
		seen[v] = struct{}{}
		(*i)[j] = v
		j++
	}
	*i = (*i)[:j]
}

// FileExists returns true if the file exists, otherwise it returns false
func FileExists(path string) (bool, error) {
	_, err := os.Stat(path)
	if err == nil {
		return true, nil
	}
	if os.IsNotExist(err) {
		return false, nil
	}
	return true, err
}

// DeepCompare compares the contents of two files at the given paths `path1` and `path2`
func DeepCompare(path1, path2 string) (bool, error) {
	file1, err := os.Open(path1)
	if err != nil {
		return false, err
	}
	defer func(f *os.File) {
		err := f.Close()
		if err != nil {
			fmt.Println("Unhandled error in DeepCompare file1.Close():", err)
		}
	}(file1)

	file2, err := os.Open(path2)
	if err != nil {
		return false, err
	}
	defer func(f *os.File) {
		err := f.Close()
		if err != nil {
			fmt.Println("Unhandled error in DeepCompare file2.Close():", err)
		}
	}(file2)

	file1Scan := bufio.NewScanner(file1)
	file2Scan := bufio.NewScanner(file2)

	for file1Scan.Scan() {
		file2Scan.Scan()
		if !bytes.Equal(file1Scan.Bytes(), file2Scan.Bytes()) {
			return false, nil
		}
	}

	return true, nil
}

// GetKeys returns a slice of keys from a map with string keys
func GetKeys(myMap map[string]string) []string {
	keys := make([]string, len(myMap))

	i := 0
	for k := range myMap {
		keys[i] = k
		i++
	}

	return keys
}

// GetInputString returns the string input by the user
func GetInputString() (string, error) {
	reader := bufio.NewReader(os.Stdin)
	return reader.ReadString('\n')
}

// RemoveFiles takes a list of files to remove using os.Remove
func RemoveFiles(files []string) {
	for _, file := range files {
		// We don't care about the error value here. If it fails, there's nothing we can do
		_ = os.Remove(file)
	}
}
