package vapour

import (
	"bytes"
	"fmt"
	"io"
	"io/ioutil"
	"os"
	"path/filepath"
	"strconv"
	"strings"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/s3"

	gpath "github.com/Masterminds/glide/path"
)

// S3Service is a wrapper class around aws-sdk-go's S3-service with a few extras
type S3Service struct {
	*s3.S3

	bucketName           string
	prefix               string
	currentVersionS3Path string
}

// CreateService initializes S3 service and wraps it in our wrapper class
func CreateService(bucketName, prefix string) (*S3Service, error) {
	sess, err := session.NewSession()
	if err != nil {
		return nil, err
	}

	cred := credentials.NewEnvCredentials()
	// XXX(pajlada): Do we even need to call this?
	cred.Get()

	awsConfig := aws.Config{
		Region:      aws.String("eu-central-1"),
		Credentials: cred,
	}

	s3Service := &S3Service{
		s3.New(sess, &awsConfig),
		bucketName,
		prefix,
		"/release/" + prefix + "/currentVersion.txt",
	}

	return s3Service, nil
}

// GetLatestVersion polls s3's currentVersion.txt of the given project and returns
// what the latest available is
// Returns 0 on errors (i.e. if no currentVersion.txt exists)
func (svc *S3Service) GetLatestVersion() int {
	goi := s3.GetObjectInput{
		Bucket: aws.String(svc.bucketName),
		Key:    aws.String(svc.currentVersionS3Path),
	}

	res, err := svc.GetObject(&goi)
	if err != nil {
		return 0
	}

	b, err := ioutil.ReadAll(res.Body)
	if err != nil {
		return 0
	}

	latestVersionString := strings.Trim(string(b), "\n\r ")

	ver, err := strconv.Atoi(latestVersionString)
	if err != nil {
		return 0
	}

	return ver
}

// DownloadTemporary downloads a file from s3 to a temporary file path
// Return the teporary file path
func (svc *S3Service) DownloadTemporary(s3Path string) (string, error) {
	goi := s3.GetObjectInput{
		Bucket: aws.String(svc.bucketName),
		Key:    aws.String(s3Path),
	}

	res, err := svc.GetObject(&goi)
	if err != nil {
		return "", err
	}

	tmpFile, err := ioutil.TempFile("", svc.prefix)
	if err != nil {
		return "", err
	}

	defer func(f *os.File) {
		err := f.Close()
		if err != nil {
			fmt.Println("Unhandled error in DownloadTemporary tmpFile.Close():", err)
		}
	}(tmpFile)

	_, err = io.Copy(tmpFile, res.Body)

	return tmpFile.Name(), nil

}

// Polls s3 for the changes of the given version
func (svc *S3Service) getChangesFor(version int, changes map[string]string) error {
	goi := s3.GetObjectInput{
		Bucket: aws.String(svc.bucketName),
		Key:    aws.String("/release/" + svc.prefix + "/versions/" + strconv.Itoa(version) + ".txt"),
	}

	res, err := svc.GetObject(&goi)
	if err != nil {
		return err
	}

	b, err := ioutil.ReadAll(res.Body)
	if err != nil {
		return err
	}

	changesStr := string(b)
	changesList := strings.Split(changesStr, "\n")

	for _, changedString := range changesList {
		changedList := strings.Split(changedString, ":")
		if len(changedList) == 2 {
			changes[changedList[0]] = strings.Trim(changedList[1], " \r\n")
		}
	}

	return nil
}

// GetChanges returns a list of file changes between currentVersion and latestVersion
func (svc *S3Service) GetChanges(currentVersion int, latestVersion int) (map[string]string, error) {
	changes := map[string]string{}
	// Figure out what updates we will need to download
	for currentVersion := currentVersion + 1; currentVersion <= latestVersion; currentVersion++ {
		err := svc.getChangesFor(currentVersion, changes)
		if err != nil {
			return nil, err
		}
	}

	return changes, nil
}

// DownloadChanges downloads a list of changes from s3 to temporary paths
// The argument is a map with s3 paths pointing to local paths
// Return map with temporary paths pointing to their end path
func (svc *S3Service) DownloadChanges(changes map[string]string) (map[string]string, error) {
	tmpPaths := map[string]string{}
	for s3Path, localPath := range changes {
		tmpPath, err := svc.DownloadTemporary(s3Path)
		if err != nil {
			return nil, err
		}
		tmpPaths[tmpPath] = localPath
	}

	return tmpPaths, nil
}

// MoveTemporaryFiles moves a set of files from A to B
// The maps keys are the temporary paths and the values are the local "end" paths
func MoveTemporaryFiles(tmpPaths map[string]string) error {
	defer func() {
		// Remove all temporary files if we fail
		RemoveFiles(GetKeys(tmpPaths))
	}()

	for tmpPath, localPath := range tmpPaths {
		// Create required folders
		err := os.MkdirAll(filepath.Dir(localPath), os.ModeDir)
		if err != nil {
			return err
		}
		err = os.Rename(tmpPath, localPath)
		if err != nil {
			err = gpath.CopyFile(tmpPath, localPath)
			if err != nil {
				return err
			}
		}
	}

	return nil
}

// GetCurrentVersion reads currentVersion.txt and returns the current version in the file
func GetCurrentVersion(path string) int {
	currentVersionBytes, err := ioutil.ReadFile(path)
	if err != nil {
		return 0
	}

	currentVersionString := strings.Trim(string(currentVersionBytes), "\n\r ")

	ver, err := strconv.Atoi(currentVersionString)
	if err != nil {
		return 0
	}

	return ver
}

// UpdateCurrentVersion updates currentVersion.txt with the `newVersion` value
func UpdateCurrentVersion(path string, newVersion int) error {
	data := strconv.Itoa(newVersion)
	err := ioutil.WriteFile(path, []byte(data), 0)
	return err
}

// GenerateS3Path generates an S3 path from a local path
// a turns into /release/TARGET/a
func (svc *S3Service) GenerateS3Path(localPath string) string {
	return "/release/" + svc.prefix + "/" + localPath
}

// GenerateChanges generates a change map from a list of changed files
// a,b,c turns into s3://a=a, s3://b=b etc
func (svc *S3Service) GenerateChanges(changedFiles []string) (map[string]string, error) {
	changes := map[string]string{}

	for _, localPath := range changedFiles {
		exists, _ := FileExists(localPath)
		if !exists {
			return nil, fmt.Errorf("File %s does not exist", localPath)
		}
		s3Path := svc.GenerateS3Path(localPath)
		changes[s3Path] = localPath
	}

	return changes, nil
}

// UploadReader uploads data from a reader to an s3 path
func (svc *S3Service) UploadReader(reader io.ReadSeeker, s3Path string) error {
	poi := s3.PutObjectInput{
		Bucket: aws.String(svc.bucketName),
		Key:    aws.String(s3Path),
		Body:   reader,
	}

	_, err := svc.PutObject(&poi)
	if err != nil {
		return err
	}

	return nil
}

// UploadFile uploads data from a local path to an s3 path
func (svc *S3Service) UploadFile(localPath, s3Path string) error {
	fileBytes, err := ioutil.ReadFile(localPath)
	if err != nil {
		return err
	}

	return svc.UploadReader(bytes.NewReader(fileBytes), s3Path)
}

// UploadChanges uploads all changed files in the given `changes` map to s3
// Map format: key=s3 path, value=localPath
func (svc *S3Service) UploadChanges(changes map[string]string) error {
	for s3Path, localPath := range changes {
		err := svc.UploadFile(localPath, s3Path)
		if err != nil {
			return err
		}
	}
	return nil
}

// UploadVersionFile generates a changes file for the given version with a map of files that were changed in that version.
// It then uploads it to the relevant versions file
func (svc *S3Service) UploadVersionFile(changes map[string]string, newVersion int) error {
	dataBytes := []byte{}
	for s3Path, localPath := range changes {
		byteArray := []byte(fmt.Sprintf("%s:%s\n", s3Path, localPath))
		dataBytes = append(dataBytes, byteArray...)
	}

	s3Path := fmt.Sprintf("/release/%s/versions/%d.txt", svc.prefix, newVersion)

	return svc.UploadReader(bytes.NewReader(dataBytes), s3Path)
}

// UpdateLatestVersion updates the latest version on s3 for target to `newVersion`
func (svc *S3Service) UpdateLatestVersion(newVersion int) error {
	return svc.UploadReader(strings.NewReader(strconv.Itoa(newVersion)), svc.currentVersionS3Path)
}
