#pragma once

#include "debug/log.hpp"

#include <boost/any.hpp>

#include <vector>

namespace Vape {

namespace detail {

#define XD_NUMBER(t, x)                         \
    if (in.type() == typeid(x)) {               \
        return static_cast<t>(any_cast<x>(in)); \
    }

template <typename NumberType>
NumberType
ParseNumber(const boost::any &in)
{
    using boost::any_cast;

    XD_NUMBER(NumberType, float)
    XD_NUMBER(NumberType, double)
    XD_NUMBER(NumberType, uint8_t)
    XD_NUMBER(NumberType, uint16_t)
    XD_NUMBER(NumberType, uint32_t)
    XD_NUMBER(NumberType, uint64_t)
    XD_NUMBER(NumberType, int8_t)
    XD_NUMBER(NumberType, int16_t)
    XD_NUMBER(NumberType, int32_t)
    XD_NUMBER(NumberType, int64_t)

    return NumberType();
}

}  // namespace detail

template <typename Type>
struct ParseAny {
    static Type
    get(const boost::any &in)
    {
        using boost::any_cast;

        if (in.empty()) {
            return Type();
        }

        if (in.type() == typeid(Type)) {
            return any_cast<Type>(in);
        }

        Log::Println("Unknown or bad type");
        return Type();
    }
};

#define ANY_NUMBER(x)                          \
    template <>                                \
    struct ParseAny<x> {                       \
        static x                               \
        get(const boost::any &in)              \
        {                                      \
            using boost::any_cast;             \
                                               \
            if (in.empty()) {                  \
                return x();                    \
            }                                  \
                                               \
            return detail::ParseNumber<x>(in); \
        }                                      \
    };

ANY_NUMBER(float)
ANY_NUMBER(double)
ANY_NUMBER(uint8_t)
ANY_NUMBER(uint16_t)
ANY_NUMBER(uint32_t)
ANY_NUMBER(uint64_t)
ANY_NUMBER(int8_t)
ANY_NUMBER(int16_t)
ANY_NUMBER(int32_t)
ANY_NUMBER(int64_t)

template <typename... Args>
typename std::enable_if<sizeof...(Args) == 1, void>::type
ParseAnyArgsToTuple(std::tuple<Args...> &arguments, const std::vector<boost::any> &vec)
{
    std::get<0>(arguments) =
        ParseAny<std::remove_reference_t<decltype(std::get<0>(arguments))>>::get(vec[0]);
}

template <typename... Args>
typename std::enable_if<sizeof...(Args) == 2, void>::type
ParseAnyArgsToTuple(std::tuple<Args...> &arguments, const std::vector<boost::any> &vec)
{
    std::get<0>(arguments) =
        ParseAny<std::remove_reference_t<decltype(std::get<0>(arguments))>>::get(vec[0]);
    std::get<1>(arguments) =
        ParseAny<std::remove_reference_t<decltype(std::get<1>(arguments))>>::get(vec[1]);
}

template <typename... Args>
typename std::enable_if<sizeof...(Args) == 3, void>::type
ParseAnyArgsToTuple(std::tuple<Args...> &arguments, const std::vector<boost::any> &vec)
{
    std::get<0>(arguments) =
        ParseAny<std::remove_reference_t<decltype(std::get<0>(arguments))>>::get(vec[0]);
    std::get<1>(arguments) =
        ParseAny<std::remove_reference_t<decltype(std::get<1>(arguments))>>::get(vec[1]);
    std::get<2>(arguments) =
        ParseAny<std::remove_reference_t<decltype(std::get<2>(arguments))>>::get(vec[2]);
}

template <typename... Args>
typename std::enable_if<sizeof...(Args) == 4, void>::type
ParseAnyArgsToTuple(std::tuple<Args...> &arguments, const std::vector<boost::any> &vec)
{
    std::get<0>(arguments) =
        ParseAny<std::remove_reference_t<decltype(std::get<0>(arguments))>>::get(vec[0]);
    std::get<1>(arguments) =
        ParseAny<std::remove_reference_t<decltype(std::get<1>(arguments))>>::get(vec[1]);
    std::get<2>(arguments) =
        ParseAny<std::remove_reference_t<decltype(std::get<2>(arguments))>>::get(vec[2]);
    std::get<3>(arguments) =
        ParseAny<std::remove_reference_t<decltype(std::get<3>(arguments))>>::get(vec[3]);
}

}  // namespace Vape
