#pragma once

#include <algorithm>
#include <chrono>
#include <cmath>
#include <cstdio>
#include <random>

namespace Vape {

class Backoff
{
    using DurationType = std::chrono::milliseconds;

    const DurationType minimumDuration;
    const DurationType maximumDuration;
    const float factor;
    const bool jitter;

    std::random_device rd;
    std::mt19937 e2;
    std::uniform_real_distribution<> dist{0.0F, 1.0F};

    int step{1};

public:
    // Available parameters:
    // minimumDuration: The minimum backoff duration
    // maximumDuration: The maximum backoff duration
    // factor: The factor at which the duration is increased each step (minDuration * (step ^ factor))
    // jitter: Enables jitter in the duration calculation
    Backoff(DurationType _minimumDuration, DurationType _maximumDuration, float _factor,
            bool _jitter)
        : minimumDuration(_minimumDuration)
        , maximumDuration(_maximumDuration)
        , factor(_factor)
        , jitter(_jitter)
        , e2(this->rd())
    {
    }

    // Step returns the current backoff step. Starts at 0
    // Can be used for checking if this is our first time attempting something
    int
    Step() const
    {
        return this->step;
    }

    // Duration returns the current duration the backoff is at now
    // In case jitter is enabled, it will also add 0-minDuration to this return value
    // Return value is clamped between 0-maxDuration
    // After the current duration has been calculated, increase step by one for the next calculation
    DurationType
    Duration()
    {
        const auto realf = std::pow(this->step, this->factor);
        DurationType duration(std::llround(this->minimumDuration.count() * realf));

        if (this->jitter) {
            auto randomValue = this->dist(this->e2);
            DurationType jitterDuration(std::llround(this->minimumDuration.count() * randomValue));
            duration += jitterDuration;
        }

        ++this->step;

        return std::min(duration, this->maximumDuration);
    }

    // Reset resets the current backoff stepping mechanism
    // Example usage: When you successfully establish a connection to a server
    void
    Reset()
    {
        this->step = 1;
    }
};

}  // namespace Vape
