#pragma once

#include <filesystem>
#include <string>

#if defined(WIN32)
#include <shlobj.h>

#include <codecvt>
#include <locale>
#else
#include <cstdlib>
#endif

namespace Vape {

namespace detail {

static inline std::string
GetConfigFolder(const std::string &projectName)
{
#if defined(WIN32)
    LPWSTR wszPath = nullptr;

    if (!SUCCEEDED(SHGetKnownFolderPath(FOLDERID_RoamingAppData, 0, nullptr, &wszPath))) {
        throw std::runtime_error("GetKnownFolderPath failed");
    }

    std::filesystem::path realPath(wszPath);

    realPath.append(projectName);

    return realPath.string();
#else
    const char *home_p = std::getenv("HOME");

    return std::string(home_p) + "/.config/" + projectName;
#endif
}

}  // namespace detail

static inline std::string
GetConfigFolder(const std::string &projectName)
{
    auto configFolder = detail::GetConfigFolder(projectName);

    std::filesystem::create_directories(configFolder);

    if (!std::filesystem::is_directory(configFolder)) {
        throw std::runtime_error("Could not create config folder");
    }

    return configFolder;
}

static inline std::string
GetConfigPath(const std::string &projectName, const std::string &fileName)
{
    auto configFolder = detail::GetConfigFolder(projectName);

    std::filesystem::create_directories(configFolder);

    if (!std::filesystem::is_directory(configFolder)) {
        throw std::runtime_error("Could not create config folder");
    }

    std::filesystem::path configPath(configFolder);
    configPath.append(fileName);

    return configPath.string();
}

}  // namespace Vape
