#pragma once

#include "fundamentals/deserializable.hpp"
#include "fundamentals/objectstore.hpp"

#include <rapidjson/document.h>

namespace Vape {

struct PendingValue {
    PendingValue()                          = delete;
    PendingValue(const PendingValue &other) = delete;
    PendingValue &operator=(const PendingValue &other) = delete;

    PendingValue(PendingValue &&other) noexcept
        : value(std::move(other.value))
        , userData(std::move(other.userData))
        , holder(std::move(other.holder))
    {
    }

    PendingValue &
    operator=(PendingValue &&other) noexcept
    {
        this->value    = std::move(other.value);
        this->userData = std::move(other.userData);
        this->holder   = std::move(other.holder);

        return *this;
    }

    PendingValue(rapidjson::Value &&_value, boost::any &&_userData,
                 std::shared_ptr<rapidjson::Document> &&_holder)
        : value(std::move(_value))
        , userData(std::move(_userData))
        , holder(std::move(_holder))
    {
    }

    rapidjson::Value value;

    boost::any userData;

    // holder is kept alive so value can stay alive
    std::shared_ptr<rapidjson::Document> holder;
};

class DeserializableStore : public ObjectStore<IDeserializable>
{
    DeserializableStore();

    // TODO: Make a function to clear this.
    // It should be cleared whenever Core is disconnected
    std::map<std::string, PendingValue> pendingValues;

public:
    static DeserializableStore &
    deserializableInstance()
    {
        static DeserializableStore i;

        return i;
    }

    enum UpdateReturnValue {
        Success,
        ObjectDoesNotExist,
        RemoteReadOnlyError,
    };

    // Update the deserializable values at the path
    DeserializableStore::UpdateReturnValue UpdateValue(const std::string &path,
                                                       const rapidjson::Value &value,
                                                       bool honorRemoteReadOnly = true,
                                                       boost::any userData = boost::any()) const;

    // Update the inner member of one of the deserializable values
    DeserializableStore::UpdateReturnValue UpdateMemberValue(
        const std::string &path, const std::string &member, const rapidjson::Value &value,
        bool honorRemoteReadOnly = true, boost::any userData = boost::any()) const;
};

}  // namespace Vape
