#pragma once

#include <cstdint>
#include <string>

namespace Vape::Environment {

/**
 * ParseString takes an input environment variable and returns its value as a string.
 * If the environment variable is unset, the given default value is returned.
 * Note, an environment variable can be set to an empty value
 **/
std::string ParseString(const char *environmentVariable, std::string defaultValue = std::string());

/**
 * ParsePort takes an input environment variable and tries to parse its string value as an unsigned 16-bit integer (a port).
 * If the environment variable is unset, the given default value is returned.
 * If the value is unable to parsed into an uint16_t, the given default value is returned.
 * If the parsed value is higher than MAX_PORT_VALUE (65535), the given default value is returned.
 **/
uint16_t ParsePort(const char *environmentVariable, const uint16_t &defaultValue);

/**
 * ParseInt takes an input environment variable and tries to parse its string value into an integer.
 * If the environment variable is unset, the given default value is returned.
 * If the value is unable to be parsed into an int, the given default value is returned.
 **/
int ParseInt(const char *environmentVariable, const int &defaultValue);

/**
 * ParseInt takes an input environment variable and tries to parse its string value into a bool.
 * If the environment variable is unset, the given default value is returned.
 * If the value matches any of the following values, return true.
 *  - 1
 * If none of the above values were matched, return false.
 **/
bool ParseBool(const char *environmentVariable, const bool &defaultValue);

}  // namespace Vape::Environment
