#pragma once

#include <fmt/format.h>
#include <fmt/ostream.h>

#include <algorithm>
#include <type_traits>
#include <vector>

namespace Vape {

template <typename T>
inline T
Clamp01(T f)
{
    return std::clamp(f, (T)0, (T)1);
}

template <typename T>
inline T
ClampEnum(const int f, const T count)
{
    return static_cast<T>(std::clamp(f, 0, count - 1));
}

template <typename T>
inline bool
EnumInRange(const T v, const T a, const T b)
{
    return v >= a && v <= b;
}

template <typename T, std::size_t N>
constexpr std::size_t
ArrayCount(const T (&/*unused*/)[N])
{
    return N;
}

template <typename T>
T
Lerp(const T &a, const T &b, const float d)
{
    float f = Clamp01(d);

    return (1.0f - f) * a + f * b;
}

// #define fS(format, ...) fmt::format(fmt(format), __VA_ARGS__)

template <typename... Args>
std::string
fS(const std::string &formatString, const Args &... args)
{
    return fmt::format(formatString, args...);
}

/*
template <typename... Args>
auto
fS(Args &&... args) -> decltype(fmt::format(std::forward<Args>(args)...))
{
    return fmt::format(std::forward<Args>(args)...);
}
*/

template <typename... Args>
constexpr std::array<std::string, sizeof...(Args)>
sA(Args &&... args)
{
    return std::array<std::string, sizeof...(Args)>{{args...}};
}

template <typename T, typename F, typename R = typename std::result_of<F &(T)>::type>
std::vector<R>
JSMap(const std::vector<T> &c, F f)
{
    std::vector<R> v;
    v.reserve(c.size());
    for (const auto &e : c) {
        v.push_back(f(e));
    }
    return v;
}

template <typename T, typename... Args>
void
SyncVectorToLength(std::vector<std::unique_ptr<T>> &vec, int length, Args... args)
{
    if (length < 0) {
        length = 0;
    }

    while (vec.size() != length) {
        if (vec.size() > length) {
            // We do this instead of vec.erase() as that creates a need for a copy constructor
            vec.pop_back();
        } else {
            vec.push_back(std::make_unique<T>(args..., vec.size()));
        }
    }
}

template <typename ContainerT, typename PredicateT>
void
EraseIf(ContainerT &items, const PredicateT &predicate)
{
    for (auto it = items.begin(); it != items.end();) {
        if (predicate(*it)) {
            it = items.erase(it);
        } else {
            ++it;
        }
    }
}

}  // namespace Vape
