#pragma once
#include <algorithm>
#include <deque>
#include <shared_mutex>

namespace Vape {
/**
 * A mutex protected deque
 */
template <typename T>
class LockedDeque
{
    std::shared_mutex mutex;
    std::deque<T> items;

public:
    LockedDeque() = default;

    /**
     * Destructor
     * Clears the deque with the mutex locked to ensure element destructors operate correctly
     */
    ~LockedDeque();

    LockedDeque(const LockedDeque<T> &) = delete;
    LockedDeque(LockedDeque<T> &&);

    /**
     * Push value to the front of the deque
     * @return Size of deque
     */
    size_t PushFront(T &&item);

    /**
     * Emplace element in place with the forwarded arguments to the front of the deque
     * @return Size of deque
     */
    template <typename... Args>
    size_t EmplaceFront(const Args &... args);

    /**
     * Push value to the back of the deque
     * @return Size of deque
     */
    size_t PushBack(T &&item);

    /**
     * Emplace element in place with the forwarded arguments to the back of the deque
     * @return Size of deque
     */
    template <typename... Args>
    size_t EmplaceBack(const Args &... args);

    /**
     * Remove elements matching value from deque
     */
    size_t RemoveValue(const T &item);

    /**
     * Remove elements where the lamba returns true
     * @return Size of deque after remove
     */
    template <typename L>
    size_t RemoveIf(L fn);

    /**
     * Clear deque
     */
    void Clear();

    /**
     * Count of items in deque
     */
    size_t Count();

    /**
     * Get copy of deque
     */
    std::deque<T> Copy();

    /**
     * Call lambda once for each element in the deque
     */
    template <typename L>
    void ForEach(L fn);

    /**
     * Call lambda once for each element, then clear deque
     * @return Number of elements before clear
     */
    template <typename L>
    size_t ForEachAndClear(L fn);

    /**
     * Call lambda with first element
     * @return Number of elements in deque
     */
    template <typename L>
    size_t Front(L fn);

    /**
     * Call lambda with last element
     * @return Number of elements in deque
     */
    template <typename L>
    size_t Back(L fn);

    /**
     * Call lambda with first element and then pop first element
     * @return Number of elements left in deque
     */
    template <typename L>
    size_t PopFront(L fn);

    /**
     * Call lambda with last element and then pop last element
     * @return Number of elements left in deque
     */
    template <typename L>
    size_t PopBack(L fn);
};

template <typename T>
LockedDeque<T>::LockedDeque(LockedDeque<T> &&o)
{
    std::unique_lock lock(this->mutex);
    std::unique_lock olock(o.mutex);

    this->items = std::move(o.items);
}

template <typename T>
LockedDeque<T>::~LockedDeque()
{
    this->Clear();
}

template <typename T>
size_t
LockedDeque<T>::PushFront(T &&item)
{
    std::unique_lock lock(this->mutex);
    this->items.push_front(std::move(item));

    return this->items.size();
}

template <typename T>
template <typename... Args>
size_t
LockedDeque<T>::EmplaceFront(const Args &... args)
{
    std::unique_lock lock(this->mutex);
    this->items.emplace_front(args...);

    return this->items.size();
}

template <typename T>
size_t
LockedDeque<T>::PushBack(T &&item)
{
    std::unique_lock lock(this->mutex);
    this->items.push_back(std::move(item));

    return this->items.size();
}

template <typename T>
template <typename... Args>
size_t
LockedDeque<T>::EmplaceBack(const Args &... args)
{
    std::unique_lock lock(this->mutex);
    this->items.emplace_back(args...);

    return this->items.size();
}

template <typename T>
size_t
LockedDeque<T>::RemoveValue(const T &item)
{
    std::unique_lock lock(this->mutex);
    if (!this->items.empty()) {
        this->items.erase(std::remove(this->items.begin(), this->items.end(), item));
    }

    return this->items.size();
}

template <typename T>
template <typename L>
size_t
LockedDeque<T>::RemoveIf(L fn)
{
    std::unique_lock lock(this->mutex);

    if (!this->items.empty()) {
        this->items.erase(std::remove_if(this->items.begin(), this->items.end(), fn),
                          this->items.end());
    }

    return this->items.size();
}

template <typename T>
void
LockedDeque<T>::Clear()
{
    std::unique_lock lock(this->mutex);
    this->items.clear();
}

template <typename T>
size_t
LockedDeque<T>::Count()
{
    std::shared_lock lock(this->mutex);
    return this->items.size();
}

template <typename T>
std::deque<T>
LockedDeque<T>::Copy()
{
    std::shared_lock lock(this->mutex);
    auto copy = this->items;
    return copy;
}

template <typename T>
template <typename L>
void
LockedDeque<T>::ForEach(L fn)
{
    std::shared_lock lock(this->mutex);
    for (auto &item : this->items) {
        fn(item);
    }
}

template <typename T>
template <typename L>
size_t
LockedDeque<T>::ForEachAndClear(L fn)
{
    std::unique_lock lock(this->mutex);
    auto size = this->items.size();

    for (auto &item : this->items) {
        fn(item);
    }

    return size;
}

template <typename T>
template <typename L>
size_t
LockedDeque<T>::Front(L fn)
{
    std::shared_lock lock(this->mutex);

    if (this->items.empty()) {
        return 0;
    }

    fn(this->items.front());
    return this->items.size();
}

template <typename T>
template <typename L>
size_t
LockedDeque<T>::Back(L fn)
{
    std::shared_lock lock(this->mutex);

    if (this->items.empty()) {
        return 0;
    }

    fn(this->items.back());
    return this->items.size();
}

template <typename T>
template <typename L>
size_t
LockedDeque<T>::PopFront(L fn)
{
    std::unique_lock lock(this->mutex);

    if (this->items.empty()) {
        return 0;
    }

    fn(this->items.front());
    this->items.pop_front();
    return this->items.size();
}

template <typename T>
template <typename L>
size_t
LockedDeque<T>::PopBack(L fn)
{
    std::unique_lock lock(this->mutex);

    if (this->items.empty()) {
        return 0;
    }

    fn(this->items.back());
    this->items.pop_back();
    return this->items.size();
}
}  // namespace Vape
